/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.views;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * Sorter f&uuml;r Benutzerliste.
 *
 * @author BitCtrl Systems GmbH, Steffen Gieseler
 *
 */
public class BenutzerListeSorter implements IBenutzerSorter, Comparator<Object> {

	/**
	 * Richtungs-Enum.
	 *
	 * @author BitCtrl Systems GmbH, Steffen Gieseler
	 */
	public enum Direction {

		/** aufsteigend. */
		ASCENDING(1),

		/** absteigend. */
		DESCENDING(-1);

		/** Wert. */
		private int value;

		/**
		 * Privater Konstruktor.
		 *
		 * @param value Wert
		 */
		Direction(final int value) {
			this.value = value;
		}

		/**
		 * Gibt den Wert zuR66uml;ck.
		 *
		 * @return Wert
		 */
		public int getValue() {
			return value;
		}

		/**
		 * Gibt die umgekehrte Richtung zur&uuml;ck.
		 *
		 * @return umgekehrte Richtung
		 */
		public Direction getInvert() {
			if (this == ASCENDING) {
				return DESCENDING;
			}
			return ASCENDING;
		}
	}

	private List<Integer> columnsList;

	private Direction[] directions;

	private final Map<Integer, IBenutzerSorter> sorterMap;
	private final Map<Integer, Direction> defaultDirectionMap;
	private final Map<Integer, Integer> columnPriorityMap;
	private int maxColumnNumber;

	private Integer[] defaultColumnSorting;
	private Direction[] defaultDirections;

	/**
	 * The default constructor.
	 *
	 * @param reset Reset
	 */
	public BenutzerListeSorter(final boolean reset) {
		if (reset) {
			resetState();
		}
		sorterMap = new HashMap<>();
		defaultDirectionMap = new HashMap<>();
		columnPriorityMap = new HashMap<>();
	}

	/**
	 * Resets the sorter to the default sorting state.
	 */
	public void resetState() {
		if ((defaultColumnSorting == null) || (defaultDirections == null)) {
			determineDefaultColumnSorting();
			determineDefaultDirections();
		}
		directions = new Direction[getDefaultDirections().length];
		System.arraycopy(getDefaultDirections(), 0, directions, 0, directions.length);
		columnsList = new ArrayList<>(Arrays.asList(getDefaultColumnSorting()));
	}

	/**
	 * Sets the column which should be sorted. If it is the same column as before,
	 * the direction will be reversed.
	 *
	 * @param column the column
	 */
	public void setColumn(final int column) {
		try {
			if (this.columnsList.get(0) != column) {
				columnsList.remove((Integer) column);
				columnsList.add(0, column);
				setDirection(column, Direction.ASCENDING);
			} else {
				reverseDirection(column);
			}
		} catch (final Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * Reverses the direction of the sorting.
	 *
	 * @param column    the column to be sorted
	 * @param direction one of the constants ASCENDING or DESCENDING
	 */
	private void setDirection(final int column, final Direction direction) {
		directions[column] = direction;
	}

	/**
	 * Returns the direction of the given column.
	 *
	 * @param column the column number
	 * @return the direction of the given column
	 */
	public Direction getDirection(final int column) {
		return directions[column];
	}

	/**
	 * Returns the current column.
	 *
	 * @return the current column
	 */
	public int getColumn() {
		return columnsList.get(0);
	}

	/**
	 * Reverses the sorting direction of the column specified.
	 *
	 * @param column the column whose sorting direction should be reversed.
	 */
	private void reverseDirection(final int column) {
		directions[column] = directions[column].getInvert();
	}

	/**
	 * Compares two objects from the table.
	 *
	 * @param e1 Objekt1
	 * @param e2 Objekt2
	 * @return 'compare'
	 */
	@Override
	public int compare(final Object e1, final Object e2) {
		for (final Integer integer : columnsList) {
			final int column = integer;
			final int result = compare(column, e1, e2);
			if (result != 0) {
				return result;
			}
		}
		return 0;
	}

	/**
	 * F&uuml;gt eine Spalte hinzu.
	 *
	 * @param column Spalte
	 * @param sorter Sorter
	 */
	public void addTableColumnSorter(final int column, final IBenutzerSorter sorter) {

		if (sorter == null) {
			return;
		}

		final Direction defaultDirection = sorter.getDefaultDirection();
		final int columnPriority = sorter.getColumnPriority();
		maxColumnNumber = (column > maxColumnNumber) ? column : maxColumnNumber;

		sorterMap.put((column < 0) ? 0 : column, sorter);
		defaultDirectionMap.put((column < 0) ? 0 : column, defaultDirection);
		columnPriorityMap.put((column < 0) ? 0 : column,
				(columnPriority < 0) ? 0 : (columnPriority > 10) ? 10 : columnPriority);
	}

	/**
	 * Bestimmt die Standard-Sortierung.
	 */
	private void determineDefaultColumnSorting() {

		final List<Entry<Integer, Integer>> entries = new ArrayList<>(columnPriorityMap.entrySet());
		Collections.sort(entries, (o1, o2) -> -1 * o1.getValue().compareTo(o2.getValue()));

		defaultColumnSorting = new Integer[entries.size()];
		for (int i = 0; i < entries.size(); i++) {
			defaultColumnSorting[i] = entries.get(i).getKey();
		}
	}

	/**
	 * Bestimmt die Standard-Richtung.
	 */
	private void determineDefaultDirections() {
		defaultDirections = new Direction[maxColumnNumber + 1];
		for (int i = 0; i <= maxColumnNumber; i++) {
			final Direction direction = defaultDirectionMap.get(i);
			defaultDirections[i] = (direction != null) ? direction : Direction.ASCENDING;
		}
	}

	/**
	 * Returns the default column sorting of this sorter.
	 *
	 * @return the default column sorting of this sorter
	 */
	protected Integer[] getDefaultColumnSorting() {
		return defaultColumnSorting;
	}

	/**
	 * Returns the default column directions of this sorter.
	 *
	 * @return the default column directions of this sorter
	 */
	protected Direction[] getDefaultDirections() {
		return defaultDirections;
	}

	/**
	 * Compares two objects from the table.
	 *
	 * @param column the column
	 * @param o1     the first object to compare
	 * @param o2     the second object to compare
	 *
	 * @return 0 for e1 = e2, negative value for e1 < e2, positive value for e1 > e2
	 */
	public int compare(final int column, final Object o1, final Object o2) {
		final IBenutzerSorter sorter = sorterMap.get(column);
		return (sorter != null) ? getDirection(column).getValue() * sorter.compare(o1, o2) : 0;
	}

	@Override
	public int getColumnPriority() {
		return 0;
	}

	@Override
	public Direction getDefaultDirection() {
		return null;
	}
}
