/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.views;

import java.util.ResourceBundle;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.window.ToolTip;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.PatternFilter;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.actions.ViewAktualisierenAktion;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.BenutzerListener;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.data.DavVerbindungSwitcher;
import de.bsvrz.buv.plugin.benutzervew.data.DavVerbindungView;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.dav.daf.main.ClientDavInterface;

/**
 * Ansicht zur Anzeige der im System definierten Nutzer und Nutzergruppen.
 *
 * @author BitCtrl Systems GmbH, Steffen Gieseler
 *
 */
public class BenutzerNavigator extends ViewPart implements BenutzerListener, DavVerbindungView {

	/** Die View-ID. */
	public static final String VIEW_ID = BenutzerNavigator.class.getName();

	/** Die Context-Hilfe-ID. */
	private static final String HILFE_ID = PluginBenutzerVew.PLUGIN_ID + "." + BenutzerNavigator.class.getSimpleName();

	/** ResourceBundle. */
	private ResourceBundle resourceBundle;

	private static final class ViewSortierenAktion extends Action {

		private final TreeViewer viewer;

		ViewSortierenAktion(final TreeViewer viewer) {
			super("Ansicht sortieren");
			setToolTipText("Sortiert die Ansicht");
			setImageDescriptor(PluginBenutzerVew.getDefault().getImageDescriptor("icons/alphab_sort_co.gif"));
			this.viewer = viewer;
		}

		@Override
		public void runWithEvent(final Event event) {
			final Tree tree = viewer.getTree();
			if (tree.getSortDirection() == SWT.DOWN) {
				tree.setSortDirection(SWT.UP);
			} else {
				tree.setSortDirection(SWT.DOWN);
			}
			viewer.refresh();
		}
	}

	/**
	 * Container.
	 */
	private Composite container;

	private TreeViewer treeViewer;

	private Text filterText;

	private DavVerbindungSwitcher verbindungsListener;

	@Override
	public void createPartControl(final Composite parent) {
		container = new Composite(parent, SWT.NONE);
		container.setLayout(new GridLayout());
//		container.addDisposeListener((event) -> dispose());

		resourceBundle = PluginBenutzerVew.getDefault().getResourceBundle();

		final FormToolkit toolkit = new FormToolkit(parent.getDisplay());

		filterText = toolkit.createText(container, "", SWT.BORDER | SWT.SINGLE);
		filterText.setMessage("Text zum Filtern eingeben");
		GridDataFactory.fillDefaults().grab(true, false).applyTo(filterText);

		treeViewer = new TreeViewer(container, SWT.SINGLE);
		treeViewer.getControl().setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true, 2, 1));

		final PatternFilter filter = new PatternFilter();
		treeViewer.addFilter(filter);

		filterText.addModifyListener((event) -> {
			filter.setPattern(filterText.getText());
			treeViewer.refresh();
		});

		treeViewer.setContentProvider(new BenutzerNavigatorContentProvider());
		treeViewer.setLabelProvider(new BenutzerNavigatorLabelProvider());

		ColumnViewerToolTipSupport.enableFor(treeViewer, ToolTip.NO_RECREATE);

		getSite().setSelectionProvider(treeViewer);

		final IActionBars bars = getViewSite().getActionBars();
		final IToolBarManager toolbar = bars.getToolBarManager();
		toolbar.add(new AllesAufklappenAction());
		toolbar.add(new AllesZusammenKlappenAction());
		toolbar.add(new ViewSortierenAktion(treeViewer));
		toolbar.add(new ViewAktualisierenAktion(treeViewer));

		createContextMenu();
		treeViewer.setComparator(new ViewerComparator() {
			@Override
			public int compare(Viewer viewer, Object e1, Object e2) {
				if (((TreeViewer) viewer).getTree().getSortDirection() == SWT.DOWN) {
					return super.compare(viewer, e2.toString().toLowerCase(), e1.toString().toLowerCase());
				}
				return super.compare(viewer, e1.toString().toLowerCase(), e2.toString().toLowerCase());
			}
		});

		// Online-Hilfe hinzufügen
		PlatformUI.getWorkbench().getHelpSystem().setHelp(treeViewer.getTree(), BenutzerNavigator.HILFE_ID);

		getSite().setSelectionProvider(treeViewer);

		verbindungsListener = new DavVerbindungSwitcher(this);
		RahmenwerkService.getService().getRahmenWerk().addDavVerbindungsListener(verbindungsListener);

		setDav(RahmenwerkService.getService().getRahmenWerk().getDavVerbindung());
		RahmenwerkService.getService().getBenutzerverwaltung().addBenutzerListener(this);
	}

	/**
	 * Erzeugt ein Kontextmenü für diese Ansicht.
	 */
	private void createContextMenu() {
		// Create menu manager.
		final MenuManager menuMgr = new BenutzerContextMenu(treeViewer);

		// Create menu.
		final Menu menu = menuMgr.createContextMenu(treeViewer.getControl());
		treeViewer.getControl().setMenu(menu);

		// Register menu for extension.
		getViewSite().registerContextMenu(menuMgr, treeViewer);
	}

	@Override
	public void setFocus() {
		if ((treeViewer != null) && (!treeViewer.getControl().isDisposed())) {
			treeViewer.getControl().setFocus();
		}
	}

	@Override
	public void dispose() {
		RahmenwerkService.getService().getRahmenWerk().removeDavVerbindungsListener(verbindungsListener);
		super.dispose();
	}

	/**
	 * liefert die ausgew&auml;hlte Nutzergruppe.
	 *
	 * @return die aktuell selektierte Nutzergruppe
	 */
	public BerechtigungsKlasse getSelectedBerechtigungsklasse() {
		if (treeViewer.getSelection() != null) {
			final Object obj = ((StructuredSelection) treeViewer.getSelection()).getFirstElement();
			if (obj instanceof BerechtigungsKlasse) {
				return (BerechtigungsKlasse) obj;
			}
		}
		return null;
	}

	public void aktualisieren(Object obj) {
		if ((treeViewer != null) && !treeViewer.getControl().isDisposed()) {
			if (obj != null) {
				treeViewer.refresh(obj);
			} else {
				treeViewer.refresh();
			}
		}
	}

	public void aktualisieren() {
		aktualisieren(null);
	}

	/**
	 * Aktion zum Zusammenklappen der gesamten Baumstruktur.
	 */
	private class AllesZusammenKlappenAction extends Action {

		AllesZusammenKlappenAction() {
			setText(resourceBundle.getString("BenutzerViewMenuCollapseAll.label"));
			setToolTipText(resourceBundle.getString("BenutzerViewMenuCollapseAll.tooltip"));
			setImageDescriptor(PluginBenutzerVew.getDefault().getImageDescriptor("icons/alles_zusammenklappen.gif"));
		}

		@Override
		public void run() {
			treeViewer.collapseAll();
		}
	}

	/**
	 * Aktion zum Aufklappen der gesamten Baumstruktur.
	 */
	private class AllesAufklappenAction extends Action {

		AllesAufklappenAction() {
			setText(resourceBundle.getString("BenutzerViewMenuExpandAll.label"));
			setToolTipText(resourceBundle.getString("BenutzerViewMenuExpandAll.tooltip"));
			setImageDescriptor(PluginBenutzerVew.getDefault().getImageDescriptor("icons/alles_aufklappen.gif"));
		}

		@Override
		public void run() {
			treeViewer.expandAll();
		}
	}

	@Override
	public void benutzerAdded(final Benutzer benutzer) {
		Display.getDefault()
				.asyncExec(() -> benutzer.getBerechtigungsKlassen().stream().forEach(klasse -> aktualisieren(klasse)));
	}

	@Override
	public void benutzerRemoved(final Benutzer benutzer) {
		Display.getDefault()
				.asyncExec(() -> benutzer.getBerechtigungsKlassen().stream().forEach(klasse -> aktualisieren(klasse)));
	}

	@Override
	public void benutzerChanged(final Benutzer benutzer) {
		Display.getDefault().asyncExec(() -> aktualisieren());
	}

	@Override
	public void setDav(ClientDavInterface dav) {
		if (dav == null) {
			setContentDescription(resourceBundle.getString("OfflineMeldung.label"));
			treeViewer.setInput(null);
			treeViewer.getControl().setEnabled(false);
		} else {
			setContentDescription("");
			treeViewer.setInput(RahmenwerkService.getService().getBenutzerverwaltung());
			treeViewer.getControl().setEnabled(true);
		}
		aktualisieren();
	}
}
