/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.views;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.resource.ColorRegistry;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.data.AngemeldeteApplikation;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.data.RolleRegionPaar;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;

/**
 * Provider zur Versorgung mit Anzeigeinformationen für Nutzer und
 * Nutzergruppen.
 *
 * @author BitCtrl Systems GmbH, Schumann
 */
public class BenutzerNavigatorLabelProvider extends CellLabelProvider implements IColorProvider, ILabelProvider {

	static {

		final ColorRegistry cr = JFaceResources.getColorRegistry();

		cr.put(BenutzerNavigatorLabelProvider.class.getName() + ".OFFLINECOLOR", new RGB(128, 128, 128));
		cr.put(BenutzerNavigatorLabelProvider.class.getName() + ".ONLINECOLOR", new RGB(0, 0, 0));
	}

	/** Die Farbe für offline Benutzer. */
	private static final Color OFFLINECOLOR = JFaceResources.getColorRegistry()
			.get(BenutzerNavigatorLabelProvider.class.getName() + ".OFFLINECOLOR");

	/** Die Farbe für online Benutzer. */
	private static final Color ONLINECOLOR = JFaceResources.getColorRegistry()
			.get(BenutzerNavigatorLabelProvider.class.getName() + ".ONLINECOLOR");

	/** Die verwendeten Images. */
	private final Map<ImageDescriptor, Image> imageCache = new HashMap<>();

	@Override
	public String getText(final Object element) {
		if (element instanceof BerechtigungsKlasse) {
			final BerechtigungsKlasse klasse = (BerechtigungsKlasse) element;
			return klasse.getName();
		} else if (element instanceof Benutzer) {
			final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();
			final Benutzer benutzer = (Benutzer) element;

			String s = benutzer.getName();

			if (bv.isAngemeldet(benutzer)) {
				s += "(angemeldet)";
			}

			return s;
		}

		return element.toString();
	}

	@Override
	public Image getImage(final Object obj) {
		ImageDescriptor descriptor = null;

		if (obj instanceof Benutzer) {
			descriptor = PluginBenutzerVew.getDefault().getImageDescriptor("icons/user.png");
		} else if (obj instanceof BerechtigungsKlasse) {
			descriptor = PluginBenutzerVew.getDefault().getImageDescriptor("icons/users.png");

		}

		Image image = null;
		if (descriptor == null) {
			final String imageKey = ISharedImages.IMG_OBJ_FILE;
			image = PlatformUI.getWorkbench().getSharedImages().getImage(imageKey);
		} else {
			image = imageCache.get(descriptor);
			if (image == null) {
				image = descriptor.createImage();
				imageCache.put(descriptor, image);
			}
		}

		return image;
	}

	@Override
	public void dispose() {
		for (final Image img : imageCache.values()) {
			img.dispose();
		}
		imageCache.clear();
	}

	@Override
	public Color getBackground(final Object element) {
		return null;
	}

	@Override
	public Color getForeground(final Object element) {
		if (element instanceof Benutzer) {
			final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();
			final Benutzer benutzer = (Benutzer) element;
			if (bv.isAngemeldet(benutzer)) {
				return ONLINECOLOR;
			}
			return OFFLINECOLOR;
		}

		return null;
	}

	@Override
	public void update(final ViewerCell cell) {
		cell.setText(getText(cell.getElement()));
		cell.setImage(getImage(cell.getElement()));
		cell.setForeground(getForeground(cell.getElement()));
		cell.setBackground(getBackground(cell.getElement()));
	}

	@Override
	public String getToolTipText(final Object element) {
		final String ls = System.getProperty("line.separator");
		if (element instanceof BerechtigungsKlasse) {
			final BerechtigungsKlasse klasse = (BerechtigungsKlasse) element;
			final StringBuffer s = new StringBuffer();
			s.append("Berechtigungsklasse: ");
			s.append(klasse.getName());
			s.append(" (");
			s.append(klasse.getPid());
			s.append(')');
			if (!klasse.getKurzInfo().isEmpty()) {
				s.append(ls);
				s.append(klasse.getKurzInfo());
			}

			final Collection<RolleRegionPaar> rolleRegionenPaare = klasse.getRolleRegionenPaare();
			if (!rolleRegionenPaare.isEmpty()) {
				s.append(ls);
				s.append("Rollen-/Regionenpaare:");
				for (final RolleRegionPaar paar : rolleRegionenPaare) {
					s.append(ls);
					s.append(paar.getRolle().getName());
					s.append('/');
					s.append(paar.getRegion().getName());
				}
			}
			return s.toString();
		} else if (element instanceof Benutzer) {
			final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();
			final Benutzer benutzer = (Benutzer) element;

			final StringBuffer s = new StringBuffer();
			s.append("Benutzer: ");
			s.append(benutzer.getName());
			s.append(" (");
			s.append(benutzer.getPid());
			s.append(')');
			if (!benutzer.getKurzInfo().isEmpty()) {
				s.append(ls);
				s.append(benutzer.getKurzInfo());
			}

			s.append(ls);
			s.append("Name: ");
			s.append(benutzer.getNachname());
			s.append(ls);
			s.append("Vorname: ");
			s.append(benutzer.getVorname());
			s.append(ls);
			s.append("Zweiter Vorname: ");
			s.append(benutzer.getZweiterVorname());
			s.append(ls);
			s.append("Organisation: ");
			s.append(benutzer.getOrganisation());
			s.append(ls);
			s.append("E-Mail: ");
			s.append(benutzer.getEmailAdresse());

			final List<AngemeldeteApplikation> anmeldungen = bv.getAnmeldungen(benutzer);
			if (!anmeldungen.isEmpty()) {
				s.append(ls);
				s.append("angemeldet:");

				for (final AngemeldeteApplikation app : anmeldungen) {
					s.append(ls);
					s.append(app.getName());
					s.append(" / online seit ");
					s.append(app.getAnmeldeZeit());
				}
			}

			return s.toString();
		}

		return element.toString();
	}

	@Override
	public int getToolTipDisplayDelayTime(final Object object) {
		return 1000;
	}
}
