/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.views;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;

import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.benutzervew.views.BenutzerListeSorter.Direction;

/**
 * Sorter f&uuml;r BenutzerNavigator.
 *
 * @author BitCtrl Systems GmbH, Steffen Gieseler
 */
public class BenutzerNavigatorSorter extends ViewerComparator implements IBenutzerSorter {

	/** Standardrichtung. */
	private final Direction defaultDirection = Direction.ASCENDING;

	/** Die aktuelle Sortierrichtung des Views. */
	private Direction richtungView;

	/** Map <Berechtigungsklasse, Direction>. */
	private final Map<BerechtigungsKlasse, Direction> sortMap = new HashMap<>();

	/**
	 * Konstruktor.
	 */
	public BenutzerNavigatorSorter() {
		richtungView = defaultDirection;
	}

	@Override
	public int compare(final Object o1, final Object o2) {
		if (o1 instanceof BerechtigungsKlasse) {
			return sortiereKlassen(o1, o2);
		}
		return sortiereKlasse(o1, o2);
	}

	/**
	 * Sortiert die Berechtigungsklassen.
	 *
	 * @param o1 Objekt1
	 * @param o2 Objekt2
	 * @return compare-Wert
	 */
	private int sortiereKlassen(final Object o1, final Object o2) {
		final Comparator<? super String> comp = getComparator();

		if ((o1 instanceof BerechtigungsKlasse) && (o2 instanceof BerechtigungsKlasse)) {
			if (richtungView == Direction.ASCENDING) {
				return comp.compare(o1.toString(), o2.toString());
			}
			return comp.compare(o2.toString(), o1.toString());
		}

		return 0;
	}

	/**
	 * Sortiert die Benutzer.
	 *
	 * @param o1 Objekt1
	 * @param o2 Objekt2
	 * @return compare-Wert
	 */
	private int sortiereKlasse(final Object o1, final Object o2) {
		final Comparator<? super String> comp = getComparator();

		if ((o1 instanceof Benutzer) && (o2 instanceof Benutzer)) {
			final Benutzerverwaltung benutzerverwaltung = RahmenwerkService.getService().getBenutzerverwaltung();
			final BerechtigungsKlasse klasse = benutzerverwaltung.getBerechtigungsklasse((Benutzer) o1);
			if (sortMap.get(klasse) == null) {
				sortMap.put(klasse, defaultDirection);
			}

			if (sortMap.get(klasse) == Direction.ASCENDING) {
				return comp.compare(o1.toString(), o2.toString());
			}
			return comp.compare(o2.toString(), o1.toString());
		}

		return 0;
	}

	/**
	 * Kehrt die Sortierreihenfolge des Views um.
	 */
	public void richtungViewUmkehren() {
		richtungView = richtungView == Direction.ASCENDING ? Direction.DESCENDING : Direction.ASCENDING;
	}

	/**
	 * Kehrt die Sortierreihenfolge einer Berechtigungsklasse um.
	 *
	 * @param klasse Berechtigungsklasse
	 */
	public void richtungKlasseUmkehren(final BerechtigungsKlasse klasse) {
		if (sortMap.get(klasse) == null) {
			sortMap.put(klasse, defaultDirection);
		} else {
			sortMap.put(klasse,
					sortMap.get(klasse) == Direction.ASCENDING ? Direction.DESCENDING : Direction.ASCENDING);
		}
	}

	@Override
	public int getColumnPriority() {
		return 0;
	}

	@Override
	public Direction getDefaultDirection() {
		return defaultDirection;
	}

	@Override
	public int compare(final Viewer viewer, final Object o1, final Object o2) {
		return compare(o1, o2);
	}
}
