/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.wizards;

import java.util.Collection;
import java.util.Map.Entry;
import java.util.SortedMap;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.BenutzerChangeException;
import de.bsvrz.buv.plugin.benutzervew.data.BenutzerInfo;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.data.KeineRechteException;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Ein Wizard zum Anlegen eines Nutzers.
 *
 * @author BitCtrl Systems GmbH, Schumann, Gieseler
 */
public class BenutzerAnlegenWizard extends Wizard implements INewWizard {

	/** Die Datenseite mit den entsprechenden Daten. */
	private BenutzerAnlegenWizardPage page;

	/** Die Datenseite mit den entsprechenden Daten. */
	private BenutzerBerechtigungsKlassenWizardPage pageKlassen;

	/** Die Datenseite mit den entsprechenden Daten. */
	private BenutzerEinmalPassworteWizardPage pageRechte;

	/** Die Ansicht, in der die Nutzer angezeigt werden. */

	/** Debug-Logger. */
	private final Debug debug = Debug.getLogger();

	/** Die ausgew&auml;hlte Berechtigungsklasse. */
	private final BerechtigungsKlasse berechtigungsKlasse;

	/**
	 * Konstruktor.
	 *
	 * @param berechtigungsKlasse Die ausgw&auml;hlte Berechtigungsklasse.
	 */
	public BenutzerAnlegenWizard(final BerechtigungsKlasse berechtigungsKlasse) {
		this.berechtigungsKlasse = berechtigungsKlasse;
		setWindowTitle("Benutzer anlegen");
	}

	@Override
	public void addPages() {
		page = new BenutzerAnlegenWizardPage(null, berechtigungsKlasse);
		addPage(page);
		pageKlassen = new BenutzerBerechtigungsKlassenWizardPage(berechtigungsKlasse);
		addPage(pageKlassen);
		pageRechte = new BenutzerEinmalPassworteWizardPage(null);
		addPage(pageRechte);
	}

	private boolean createUser() {

		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (!rahmenwerk.isOnline()) {
			return false;
		}

		final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();
		final String loginname = page.getLoginname();
		final Collection<BerechtigungsKlasse> klassen = pageKlassen.getBerechtigungsklassen();

		Assert.isNotNull(loginname);
		Assert.isTrue(loginname.length() > 0);

		final Benutzer b = bv.getBenutzer(loginname);
		if (b != null) {
			page.setErrorMessage("Der Nutzer " + loginname + " existiert bereits!");
			return false;
		}

		final BenutzerInfo benutzerInfo = new BenutzerInfo();
		benutzerInfo.setPasswort(page.getPasswort());
		benutzerInfo.setNachname(page.getNachname());
		benutzerInfo.setVorname(page.getVorname());
		benutzerInfo.setZweiterVorname(page.getZweiterVorname());
		benutzerInfo.setEmailAdresse(page.getEmailAdresse());
		benutzerInfo.setOrganisation(page.getOrganisation());

		final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(
				PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), (verbindung, urlasser) -> {
					// TODO Datenverteilerverbindung verwenden!
				});
		dialog.setInitialBenutzer(rahmenwerk.getBenutzerName());
		dialog.setInitialUrsache("Benutzer '" + loginname + "' angelegt");
		dialog.setInitialVeranlasser(rahmenwerk.getBenutzerName());

		if (dialog.open() != Window.OK) {
			return false;
		}
		final UrlasserInfo uinfo = dialog.getUrlasserInfo();

		final String adminLoginname = uinfo.getBenutzerName();
		final String adminPasswort = uinfo.getPasswort();

		try {
			final Benutzer benutzer = bv.anlegenBenutzer(adminLoginname, adminPasswort, loginname, benutzerInfo,
					page.isAdmin());
			final SortedMap<Integer, String> einmalPassworte = bv.setEinmalPassworte(adminLoginname, adminPasswort,
					loginname, pageRechte.getEinmalPassworte());
			bv.setBerechtigungsklassen(adminLoginname, adminPasswort, benutzer, klassen, uinfo);
			if (!einmalPassworte.isEmpty()) {
				final StringBuilder message = new StringBuilder();
				message.append("Für den Benutzer '");
				message.append(benutzer.getName());
				message.append("' wurden folgende Einmalpassworte angelegt.\n\n");
				for (final Entry<Integer, String> einmalPasswort : einmalPassworte.entrySet()) {
					message.append(einmalPasswort.getKey());
					message.append(" - ");
					message.append(einmalPasswort.getValue());
					message.append('\n');
				}
				MessageDialog.openInformation(getShell(), "INFORMATION", message.toString());
			}

		} catch (final KeineRechteException ex) {
			String s = ex.getLocalizedMessage();
			if (ex.getCause() != null) {
				s += " " + ex.getCause().getLocalizedMessage();
			}
			page.setErrorMessage(s);
			debug.error(s);
			return false;
		} catch (final BenutzerChangeException ex) {
			String s = ex.getLocalizedMessage();
			if (ex.getCause() != null) {
				s += " " + ex.getCause().getLocalizedMessage();
			}
			page.setErrorMessage(s);
			debug.error(s);
			MessageDialog.openError(Display.getDefault().getActiveShell(), "Fehler", s);
			return false;
		}

		return true;
	}

	@Override
	public void init(final IWorkbench workbench, final IStructuredSelection selection) {
		// nix
	}

	@Override
	public boolean performFinish() {
		return createUser();
	}
}
