/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.wizards;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.data.PasswortInfo;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;

/**
 * Die Datenseite eines Wizards zum Anlegen der Informationen zu einem Nutzer.
 * Die Seite l&auml;sst sich erst verlassen, wenn Loginname, Nachname, Vorname,
 * Passwort und Berechtigungsklasse angegeben wurden.
 * <p>
 * Im Falle des Passworts wird gepr&uuml;ft ob die Wiederholung
 * &uuml;bereinstimmt und ob das Passwort bestimmten Kriterien entspricht.
 *
 * @author BitCtrl Systems GmbH, Schumann, Gieseler
 */
public class BenutzerAnlegenWizardPage extends WizardPage {

	/** Die Context-Hilfe-ID. */
	private static final String HILFE_ID = PluginBenutzerVew.PLUGIN_ID + "."
			+ BenutzerAnlegenWizardPage.class.getSimpleName();

	/**
	 * Interner Listener f&uuml;r &Auml;nderungen.
	 *
	 * @author BitCtrl Systems GmbH, Steffen Gieseler
	 */
//	private class ChangeListener implements ModifyListener {
//
//		@Override
//		public void modifyText(final ModifyEvent e) {
//			checkInput();
//		}
//
//	}

	/** Der Benutzername. */
	private Text loginname;

	/** Der Nachname. */
	private Text nachname;

	/** Der Vorname. */
	private Text vorname;

	/** Der zweite Vorname. */
	private Text zweiterVorname;

	/** Die E-Mail-Adresse. */
	private Text emailAdresse;

	/** Die Organistation. */
	private Text organisation;

	/** Das Passwort. */
	private Text passwort;

	/** Die Wiederholung des Passwortes. */
	private Text passwortRepeat;

	/** Das Adminstrator-Flag. */
	private boolean adminFlag;

	/** Der optional zu bearbeitende Nutzer. */
	private final Benutzer benutzer;

	/** Nutzer ist editierbar? */
	private final boolean editable = true;

	/** Button f&uuml;r Admin-Flag. */
	private Button adminButton;

	/** Passworteigenschaften. */
	private final PasswortInfo passWortInfo;

	/**
	 * Konstruktor.
	 *
	 * @param benutzer der Nutzer.
	 * @param klasse   die Berechtigungsklasse.
	 */
	public BenutzerAnlegenWizardPage(final Benutzer benutzer, final BerechtigungsKlasse klasse) {
		super("Nutzer");

		this.benutzer = benutzer;

		setTitle("Geben Sie die Daten für den neuen Nutzer ein:");

		// Passwort-Eigenschaften setzen
		passWortInfo = new PasswortInfo();
		passWortInfo.setMinLaenge(RahmenwerkService.BENUTZERPASSWORT_MIN_LAENGE);
	}

	@Override
	public void createControl(final Composite parent) {

		// Online-Hilfe hinzufügen
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, BenutzerAnlegenWizardPage.HILFE_ID);

		final Composite panel = new Composite(parent, SWT.NONE);
		final GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		panel.setLayout(pageLayout);
		panel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// Loginname
		Label label = new Label(panel, SWT.NONE);
		label.setText("Nutzername:");
		loginname = new Text(panel, SWT.SINGLE | SWT.BORDER);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		loginname.setLayoutData(gridData);
		if (benutzer != null) {
			loginname.setText(benutzer.getName());
			loginname.setEnabled(false);
		}
		loginname.addModifyListener((event) -> checkInput());

		// Nachname
		label = new Label(panel, SWT.NONE);
		label.setText("Nachname:");
		nachname = new Text(panel, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		nachname.setLayoutData(gridData);
		nachname.setEnabled(editable);
		if (benutzer != null) {
			nachname.setText(benutzer.getNachname());
			nachname.setEnabled(false);
		}
		nachname.addModifyListener((event) -> checkInput());

		// Vorname
		label = new Label(panel, SWT.NONE);
		label.setText("Vorname:");
		vorname = new Text(panel, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		vorname.setLayoutData(gridData);
		vorname.setEnabled(editable);
		if (benutzer != null) {
			vorname.setText(benutzer.getVorname());
			vorname.setEnabled(false);
		}
		vorname.addModifyListener((event) -> checkInput());

		// Zweiter Vorname
		label = new Label(panel, SWT.NONE);
		label.setText("Zweiter Vorname:");
		zweiterVorname = new Text(panel, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		zweiterVorname.setLayoutData(gridData);
		zweiterVorname.setEnabled(editable);
		if (benutzer != null) {
			zweiterVorname.setText(benutzer.getZweiterVorname());
			zweiterVorname.setEnabled(false);
		}

		// E-Mail-Adresse
		label = new Label(panel, SWT.NONE);
		label.setText("E-Mail:");
		emailAdresse = new Text(panel, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		emailAdresse.setLayoutData(gridData);
		emailAdresse.setEnabled(editable);
		if (benutzer != null) {
			emailAdresse.setText(benutzer.getEmailAdresse());
			emailAdresse.setEnabled(false);
		}

		// Organisation
		label = new Label(panel, SWT.NONE);
		label.setText("Organisation:");
		organisation = new Text(panel, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		organisation.setLayoutData(gridData);
		organisation.setEnabled(editable);
		if (benutzer != null) {
			organisation.setText(benutzer.getOrganisation());
			organisation.setEnabled(false);
		}

		// Passwort
		label = new Label(panel, SWT.NONE);
		label.setText("Passwort:");
		passwort = new Text(panel, SWT.PASSWORD | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		passwort.setLayoutData(gridData);
		passwort.addModifyListener((event) -> checkInput());

		// Passwortwiederholung
		label = new Label(panel, SWT.NONE);
		label.setText("Passwort bestätigen:");
		passwortRepeat = new Text(panel, SWT.PASSWORD | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		passwortRepeat.setLayoutData(gridData);
		passwortRepeat.addModifyListener((event) -> checkInput());

		label = new Label(panel, SWT.NONE);
		label.setText("Benutzer ist Administrator:");
		adminButton = new Button(panel, SWT.CHECK);
		adminButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				adminFlag = adminButton.getSelection();
			}

		});

		if (editable) {
			checkInput();
		}

		setControl(panel);
	}

	/**
	 * Flag ob mit der Page ein Benutzer bearbeitet oder neu angelegt wird.
	 *
	 * @return {@code true}, wenn ein vorhandener Benutzer bearbeitet wird und
	 *         {@code false}, wenn ein neuer Benutzer angelegt wird.
	 */
	public boolean isBearbeiten() {
		return benutzer != null;
	}

	/**
	 * Gibt den Benutzer zur&uuml;ck.
	 *
	 * @return Benutzer
	 */
	public Benutzer getBenutzer() {
		return benutzer;
	}

	/**
	 * Liefert das eingegebene Passwort.
	 *
	 * @return Returns the passwd.
	 */
	public String getPasswort() {
		return passwort.getText();
	}

	/**
	 * Liefert das eingegebene wiederholte Passwort.
	 *
	 * @return Returns the passwdRepeat.
	 */
	public String getPasswortRepeat() {
		return passwortRepeat.getText();
	}

	/**
	 * liefert den eingegebenen Namen.
	 *
	 * @return Returns the userName.
	 */
	public String getLoginname() {
		return loginname.getText();
	}

	/**
	 * liefert den eingegebenen Vornamen.
	 *
	 * @return Returns the vorName.
	 */
	public String getVorname() {
		return vorname.getText();
	}

	/**
	 * liefert den eingegebenen Familiennamen.
	 *
	 * @return Returns the realName.
	 */
	public String getNachname() {
		return nachname.getText();
	}

	/**
	 * liefert die eingegebene Telefonnummer.
	 *
	 * @return Returns the telefon.
	 */
	public String getZweiterVorname() {
		return zweiterVorname.getText();
	}

	/**
	 * Liefert die eingegebene E-Mail-Adresse.
	 *
	 * @return Returns the email.
	 */
	public String getEmailAdresse() {
		return emailAdresse.getText();
	}

	/**
	 * Gibt die Organisation zurück.
	 *
	 * @return die Organisation.
	 */
	public String getOrganisation() {
		return organisation.getText();
	}

	/**
	 * Gibt die Information zur&uuml;ck, ob der Benutzer Administrator-Rechte
	 * besitzt.
	 *
	 * @return {@code true}, wenn der Benutzer Administrator-Rechte besitzt
	 *         {@code false}, wenn der Benutzer keine Administrator-Rechte besitzt
	 */
	public boolean isAdmin() {
		return adminFlag;
	}

	/**
	 * &Uuml;berpr&uuml;ft alle Eingaben.
	 */
	private void checkInput() {
		final String passwortCheck;
		String errorMsg = null;
		final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();

		if (getLoginname().length() == 0) {
			errorMsg = "Sie müssen einen Nutzernamen eingeben.";
		}

		// Den Teil nur prüfen beim Anlegen neuer Benutzer. Beim Bearbeiten muss
		// nicht geprüft werden, da nicht änderbar.
		if (getBenutzer() == null) {
			if ((getLoginname().length() > 0) && (bv.getBenutzer(getLoginname()) != null)) {
				errorMsg = "Es existiert bereits ein Benutzer mit dieser Kennung im System.";
			}

			if ((errorMsg == null) && (getNachname().length() == 0)) {
				errorMsg = "Sie müssen den Nachnamen eingeben.";
			} else if ((errorMsg == null) && (getVorname().length() == 0)) {
				errorMsg = "Sie müssen den Vornamen eingeben.";
			}
		}

		if (errorMsg == null) {
			passwortCheck = bv.checkPasswort(getPasswort(), null, passWortInfo);
			if (passwortCheck != null) {
				// if (getBenutzer() == null || getPasswort().length() > 0) {
				errorMsg = passwortCheck;
				// }
			}
		}

		if ((errorMsg == null) && !getPasswort().equals(getPasswortRepeat())) {
			errorMsg = "Das Passwort stimmt nicht mit der Wiederholung überein";
		}

		setMessage(errorMsg, WARNING);
		setPageComplete(errorMsg == null);
	}

}
