/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.wizards;

import java.util.Map.Entry;
import java.util.SortedMap;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.BenutzerChangeException;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.KeineRechteException;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Ein Wizard zum Bearbeiten der Berechtigungen eines Nutzers.
 *
 * @author BitCtrl Systems GmbH, Schumann, Gieseler
 */
public class BenutzerBearbeitenWizard extends Wizard implements INewWizard {

	/** Die Datenseite mit den entsprechenden Daten. */
	private BenutzerBerechtigungsKlassenWizardPage page;

	/** Die Datenseite mit den entsprechenden Daten. */
	private BenutzerEinmalPassworteWizardPage pageRechte;

	/** Der Nutzer, der bearbeitet werden soll. */
	private final Benutzer benutzer;

	/** Debug-LOgger. */
	private final Debug log = Debug.getLogger();

	/**
	 * Konstruktor.
	 *
	 * @param benutzer der Nutzer, der bearbeitet werden soll.
	 *
	 */
	public BenutzerBearbeitenWizard(final Benutzer benutzer) {
		Assert.isNotNull(benutzer, "Der Benutzer darf nicht null sein.");

		setWindowTitle("Benutzerberechtigungen bearbeiten");

		this.benutzer = benutzer;
	}

	@Override
	public void addPages() {
		page = new BenutzerBerechtigungsKlassenWizardPage(benutzer);
		addPage(page);
		pageRechte = new BenutzerEinmalPassworteWizardPage(benutzer);
		addPage(pageRechte);
	}

	/**
	 * Erzeugt einen neuen Nutzer.
	 *
	 * @return Nutzer wurde angelegt?
	 */
	private boolean changeUser() {

		final Rahmenwerk rahmenwerk = RahmenwerkService.getService().getRahmenWerk();
		if (!rahmenwerk.isOnline()) {
			return false;
		}

		final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(
				PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), (verbindung, urlasser) -> {
					// TODO Datenverteilerverbindung verwenden
				});

		dialog.setInitialBenutzer(rahmenwerk.getBenutzerName());
		dialog.setInitialUrsache("Benutzer '" + benutzer.getName() + "' bearbeitet");
		dialog.setInitialVeranlasser(rahmenwerk.getBenutzerName());

		if (dialog.open() != Window.OK) {
			return false;
		}

		final UrlasserInfo uinfo = dialog.getUrlasserInfo();

		final String adminLoginname = uinfo.getBenutzerName();
		final String adminPasswort = uinfo.getPasswort();

		final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();

		try {
			bv.setBerechtigungsklassen(adminLoginname, adminPasswort, benutzer, page.getBerechtigungsklassen(), uinfo);
			final SortedMap<Integer, String> einmalPassworte = bv.setEinmalPassworte(adminLoginname, adminPasswort,
					benutzer.getName(), pageRechte.getEinmalPassworte());
			if (!einmalPassworte.isEmpty()) {
				final StringBuilder message = new StringBuilder();
				message.append("Für den Benutzer '");
				message.append(benutzer.getName());
				message.append("' wurden folgende Einmalpassworte angelegt.\n\n");
				for (final Entry<Integer, String> einmalPasswort : einmalPassworte.entrySet()) {
					message.append(einmalPasswort.getKey());
					message.append(" - ");
					message.append(einmalPasswort.getValue());
					message.append('\n');
				}
				MessageDialog.openInformation(getShell(), "INFORMATION", message.toString());
			}

		} catch (final KeineRechteException ex) {
			String s = ex.getLocalizedMessage();
			if (ex.getCause() != null) {
				s += " " + ex.getCause().getLocalizedMessage();
			}
			page.setErrorMessage(s);
			log.error(ex.getLocalizedMessage(), ex);
			return false;
		} catch (final BenutzerChangeException ex) {
			String s = ex.getLocalizedMessage();
			if (ex.getCause() != null) {
				s += " " + ex.getCause().getLocalizedMessage();
			}
			page.setErrorMessage(s);
			log.error(ex.getLocalizedMessage(), ex);
			return false;
		}

		return true;
	}

	@Override
	public void init(final IWorkbench workbench, final IStructuredSelection selection) {
		// nix
	}

	@Override
	public boolean performFinish() {
		return changeUser();
	}
}
