/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.wizards;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzer;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.PasswortInfo;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;

/**
 * Die Datenseite eines Wizards zum &Auml;dern des Passwortes eines Nutzer.
 * <p>
 * F&uuml;r das Passworts wird gepr&uuml;ft, ob die Wiederholung
 * &uuml;bereinstimmt und ob das Passwort bestimmten Kriterien entspricht.
 *
 * @author BitCtrl Systems GmbH, Schumann, Gieseler
 */
public class BenutzerPasswortWizardPage extends WizardPage {
	/** Die Context-Hilfe-ID. */
	private static final String HILFE_ID = PluginBenutzerVew.PLUGIN_ID + "."
			+ BenutzerPasswortWizardPage.class.getSimpleName();

	/**
	 * Interner Listener f&uuml;r &Auml;nderungen.
	 *
	 * @author BitCtrl Systems GmbH, Steffen Gieseler
	 */
	private class ChangeListener implements ModifyListener {

		@Override
		public void modifyText(final ModifyEvent e) {
			checkInput();
		}
	}

	/** Textfeld aktuelles Passwort. */
	private Text aktuellesPasswort;

	/** Textfeld Passwort. */
	private Text passwort;

	/** Textfeld Passwort-Wiederholung. */
	private Text passwortRepeat;

	/** Der zu bearbeitende Nutzer. */
	private final Benutzer benutzer;

	/** Nutzer ist editierbar? */
	private final boolean editable = true;

	/** Passwort-Eigenschaften. */
	private final PasswortInfo passWortInfo;

	/** Flag, ob das eigene Passwort ge&auml;ndert werden soll. */
	private final boolean istEigenesPasswort;

	/**
	 * Konstruktor.
	 *
	 * @param benutzer der Nutzer.
	 */
	public BenutzerPasswortWizardPage(final Benutzer benutzer) {
		super("Benutzer");

		this.benutzer = benutzer;
		istEigenesPasswort = (benutzer == null)
				|| benutzer.getName().equals(RahmenwerkService.getService().getRahmenWerk().getBenutzerName());

		setTitle("Geben Sie ein neues Passwort für den Benutzer '" + (benutzer == null ? null : benutzer.getName())
				+ "' an");

		// Passwort-Eigenschaften setzen
		passWortInfo = new PasswortInfo();
		passWortInfo.setMinLaenge(RahmenwerkService.BENUTZERPASSWORT_MIN_LAENGE);
	}

	@Override
	public void createControl(final Composite parent) {

		// Online-Hilfe hinzufügen
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, BenutzerPasswortWizardPage.HILFE_ID);

		final Composite panel = new Composite(parent, SWT.NONE);
		final ChangeListener changeListener = new ChangeListener();
		final GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		panel.setLayout(pageLayout);
		panel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Label label;
		GridData gridData;

		if (istEigenesPasswort) {
			// wenn angemeldeter Benutzer: altes Passwort
			label = new Label(panel, SWT.NONE);
			label.setText("aktuelles Passwort:");
			aktuellesPasswort = new Text(panel, SWT.PASSWORD | SWT.BORDER);
			gridData = new GridData(GridData.FILL_HORIZONTAL);
			aktuellesPasswort.setLayoutData(gridData);
			aktuellesPasswort.addModifyListener(changeListener);
		}

		// Passwort
		label = new Label(panel, SWT.NONE);
		label.setText("neues Passwort:");
		passwort = new Text(panel, SWT.PASSWORD | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		passwort.setLayoutData(gridData);
		passwort.addModifyListener(changeListener);

		// Passwortwiederholung
		label = new Label(panel, SWT.NONE);
		label.setText("neues Passwort bestätigen:");
		passwortRepeat = new Text(panel, SWT.PASSWORD | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		passwortRepeat.setLayoutData(gridData);
		passwortRepeat.addModifyListener(changeListener);

		if (editable) {
			checkInput();
		}

		setControl(panel);
	}

	/**
	 * Flag ob mit der Page ein Benutzer bearbeitet oder neu angelegt wird.
	 *
	 * @return {@code true}, wenn ein vorhandener Benutzer bearbeitet wird und
	 *         {@code false}, wenn ein neuer Benutzer angelegt wird.
	 */
	public boolean isBearbeiten() {
		return benutzer != null;
	}

	/**
	 * Liefert den Benutzer.
	 *
	 * @return Benutzer.
	 */
	public Benutzer getBenutzer() {
		return benutzer;
	}

	/**
	 * Liefert das aktuelle Passwort.
	 *
	 * @return Returns the passwd.
	 */
	public String getPasswortAktuell() {
		if (aktuellesPasswort != null) {
			return aktuellesPasswort.getText();
		}

		return null;
	}

	/**
	 * Liefert das eingegebene Passwort.
	 *
	 * @return Returns the passwd.
	 */
	public String getPasswort() {
		return passwort.getText();
	}

	/**
	 * Liefert das eingegebene wiederholte Passwort.
	 *
	 * @return Returns the passwdRepeat.
	 */
	public String getPasswortRepeat() {
		return passwortRepeat.getText();
	}

	/**
	 * &Uuml;berpr&uuml;ft alle Eingaben.
	 */
	private void checkInput() {
		final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();
		String errorMsg = bv.checkPasswort(getPasswort(), null, passWortInfo);

		if ((errorMsg == null) && !getPasswort().equals(getPasswortRepeat())) {
			errorMsg = "Das neue Passwort stimmt nicht mit der Wiederholung überein";
		}

		setMessage(errorMsg, WARNING);
		setPageComplete(errorMsg == null);
	}

}
