/*
 * Rahmenwerk-Plug-in "Benutzerverwaltung/Zugriffsrechte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.benutzervew.wizards;

import java.util.Collection;
import java.util.LinkedHashSet;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.PlatformUI;

import de.bsvrz.buv.plugin.benutzervew.PluginBenutzerVew;
import de.bsvrz.buv.plugin.benutzervew.data.Benutzerverwaltung;
import de.bsvrz.buv.plugin.benutzervew.data.BerechtigungsKlasse;
import de.bsvrz.buv.plugin.benutzervew.data.Region;
import de.bsvrz.buv.plugin.benutzervew.data.Rolle;
import de.bsvrz.buv.plugin.benutzervew.data.RolleRegionPaar;
import de.bsvrz.buv.plugin.benutzervew.internal.RahmenwerkService;

/**
 * Wizard-Datenseite zum Bearbeiten der Daten einer Nutzergruppe.
 *
 * @author BitCtrl Systems GmbH, Schumann, Gieseler
 */
public class BerechtigungsklasseWizardPage extends WizardPage {

	private static final class ZuordnungsLabelProvider extends LabelProvider implements ITableLabelProvider {

		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		@Override
		public String getColumnText(Object element, int columnIndex) {
			if (element instanceof RolleRegionPaar) {
				final RolleRegionPaar paar = (RolleRegionPaar) element;
				switch (columnIndex) {
				case 0:
					return paar.getRolle().toString();
				case 1:
					return paar.getRegion().toString();
				default:
					break;
				}
			}
			return "";
		}

	}

	/** Die Context-Hilfe-ID. */
	private static final String HILFE_ID = PluginBenutzerVew.PLUGIN_ID + "."
			+ BerechtigungsklasseWizardPage.class.getSimpleName();

	/** Die Tabelle, in der die verfügbaren Nutzerrechte dargestellt werden. */
	private TableViewer zuordnungsViewer;

	/** Die Auswahlliste der Rollen. */
	private ListViewer rollenListe;

	/** Die Auswahlliste der Regionen. */
	private ListViewer regionenListe;

	/** Button 'Hinzuf&uuml;gen'. */
	private Button addButton;

	/** Button 'Entfernen'. */
	private Button removeButton;

	private final Collection<RolleRegionPaar> rollenRegionenPaare = new LinkedHashSet<>();

	/**
	 * Konstruktor.
	 *
	 * @param klasse die Berechtigungsklasse.
	 */
	public BerechtigungsklasseWizardPage(final BerechtigungsKlasse klasse) {
		super("Berechtigungsklasse");
		setTitle("Stellen Sie hier die Rollen-/Regionenpaare der Berechtigungsklasse '" + klasse.getName()
				+ "' zusammen");
		rollenRegionenPaare.addAll(klasse.getRolleRegionenPaare());
	}

	@Override
	public void createControl(final Composite parent) {
		// Online-Hilfe hinzufügen
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, BerechtigungsklasseWizardPage.HILFE_ID);

		final Benutzerverwaltung bv = RahmenwerkService.getService().getBenutzerverwaltung();

		final Composite panel = new Composite(parent, SWT.NONE);

		// Layout der WizardPage festlegen
		panel.setLayout(new GridLayout(3, false));

		Group aGruppe = new Group(panel, SWT.SHADOW_ETCHED_IN);
		aGruppe.setText("Verfügbare Rollen:");
		aGruppe.setLayout(new GridLayout());
		GridDataFactory.fillDefaults().grab(true, true).applyTo(aGruppe);

		rollenListe = new ListViewer(aGruppe, SWT.SINGLE);
		rollenListe.setContentProvider(new ArrayContentProvider());
		rollenListe.setInput(bv.getRollen());
		rollenListe.addSelectionChangedListener((event) -> updateButtons());
		GridDataFactory.fillDefaults().grab(true, true).applyTo(rollenListe.getControl());

		final Composite buttonPanel = new Composite(panel, SWT.NONE);
		buttonPanel.setLayout(new GridLayout());
		GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.CENTER).applyTo(buttonPanel);
		addButton = new Button(buttonPanel, SWT.PUSH);
		addButton.setText(" >> ");
		addButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				rollenRegionenPaare.add(getSelectedRolleRegionPaar());
				getShell().getDisplay().asyncExec(() -> zuordnungsViewer.refresh());
			}
		});

		removeButton = new Button(buttonPanel, SWT.PUSH);
		removeButton.setText(" << ");
		removeButton.setEnabled(false);
		removeButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				rollenRegionenPaare.removeAll(((IStructuredSelection) zuordnungsViewer.getSelection()).toList());
				getShell().getDisplay().asyncExec(() -> zuordnungsViewer.refresh());

			}
		});

		erzeugeRollenRegionenTabelle(panel);

		aGruppe = new Group(panel, SWT.SHADOW_ETCHED_IN);
		aGruppe.setText("Verfügbare Regionen:");
		aGruppe.setLayout(new GridLayout());
		GridDataFactory.fillDefaults().grab(true, true).applyTo(aGruppe);

		regionenListe = new ListViewer(aGruppe, SWT.SINGLE);
		regionenListe.setContentProvider(new ArrayContentProvider());
		regionenListe.setComparator(new ViewerComparator());
		regionenListe.setInput(bv.getRegionen());
		regionenListe.addSelectionChangedListener((event) -> updateButtons());
		GridDataFactory.fillDefaults().grab(true, true).applyTo(regionenListe.getControl());

		addButton.setEnabled(false);
		setControl(panel);
	}

	private void updateButtons() {
		addButton.setEnabled(!(regionenListe.getSelection().isEmpty() || rollenListe.getSelection().isEmpty()));
		removeButton.setEnabled(!zuordnungsViewer.getSelection().isEmpty());
	}

	/**
	 * Erzeugt die Tabelle der Rollen-/Regionenpaare der Klasse.
	 *
	 * @param parent Parent-Composite
	 */
	private void erzeugeRollenRegionenTabelle(final Composite parent) {
		final Group adminGruppe = new Group(parent, SWT.SHADOW_ETCHED_IN);
		GridDataFactory.fillDefaults().grab(true, true).span(1, 2).applyTo(adminGruppe);
		adminGruppe.setText("Rollen-/Regionenpaare der Berechtigungsklasse:");
		adminGruppe.setLayout(new GridLayout(1, false));

		// Tabelle mit den Rollen und Regionen anlegen
		zuordnungsViewer = new TableViewer(adminGruppe, SWT.BORDER | SWT.FULL_SELECTION);
		zuordnungsViewer.getTable().setHeaderVisible(true);
		zuordnungsViewer.getTable().setLinesVisible(true);
		GridDataFactory.fillDefaults().grab(true, true).span(1, 2).applyTo(zuordnungsViewer.getControl());

		final TableLayout tabLayout = new TableLayout();

		// 1. Spalte für Rollen
		TableColumn column = new TableColumn(zuordnungsViewer.getTable(), SWT.NONE);
		column.setText("Zugriffsrolle");
		tabLayout.addColumnData(new ColumnWeightData(1));

		// 2. Spalte für Regionen
		column = new TableColumn(zuordnungsViewer.getTable(), SWT.NONE);
		column.setText("Zugriffsregion");
		tabLayout.addColumnData(new ColumnWeightData(1));

		zuordnungsViewer.getTable().setLayout(tabLayout);

		zuordnungsViewer.setContentProvider(new ArrayContentProvider());
		zuordnungsViewer.setLabelProvider(new ZuordnungsLabelProvider());
		zuordnungsViewer.setInput(rollenRegionenPaare);
		zuordnungsViewer.addSelectionChangedListener((event) -> updateButtons());
	}

	/**
	 * Gibt das ausgew&auml;hlte Roole/Region-Paar zur&uuml;ck.
	 *
	 * @return ausgew&auml;hltes Roole/Region-Paar
	 */
	private RolleRegionPaar getSelectedRolleRegionPaar() {
		final Region region = (Region) ((IStructuredSelection) regionenListe.getSelection()).getFirstElement();
		final Rolle rolle = (Rolle) ((IStructuredSelection) rollenListe.getSelection()).getFirstElement();
		return new RolleRegionPaar(rolle, region);
	}

	/**
	 * Liefert die Paare von Rolle und Region die für die Berechtigungsklasse
	 * gewählt wurden.
	 *
	 * @return die Rolle-Region-Paare.
	 */
	public Collection<RolleRegionPaar> getRollenRegionenPaare() {
		return rollenRegionenPaare;
	}
}
