/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.actions;

import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.jface.action.StatusLineContributionItem;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;

/**
 * Basisklasse für Status Items, die Informationen zu einer Darstellung
 * anzeigen.
 *
 * <p>
 * Das Status Item meldet sich auf den aktuellen Workbench Part an und wird nur
 * aktiviert, wenn der aktuelle Part über {@link GraphicalViewer},
 * {@link ZoomManager} und {@link Darstellung} verfügt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class DarstellungStatus extends StatusLineContributionItem {

	private final class IPartListenerImplementation implements IPartListener {

		@Override
		public void partOpened(final IWorkbenchPart part) {
			updateStatus(part);
		}

		@Override
		public void partDeactivated(final IWorkbenchPart part) {
			updateStatus(part);
		}

		@Override
		public void partClosed(final IWorkbenchPart part) {
			updateStatus(part);
		}

		@Override
		public void partBroughtToTop(final IWorkbenchPart part) {
			updateStatus(part);
		}

		@Override
		public void partActivated(final IWorkbenchPart part) {
			updateStatus(part);
		}
	}

	private final IWorkbenchPage page;
	private final IPartListenerImplementation listener;
	private IWorkbenchPart currentPart;
	private GraphicalViewer viewer;
	private ZoomManager zoomManager;
	private Darstellung darstellung;

	/**
	 * Initialisietr das Status Item.
	 *
	 * @param page
	 * @param id
	 * @param charWidth
	 */
	public DarstellungStatus(final IWorkbenchPage page, final String id, final int charWidth) {
		super(id, charWidth);
		this.page = page;

		listener = new IPartListenerImplementation();
		page.addPartListener(listener);
	}

	private void updateStatus(final IWorkbenchPart part) {
		final IEditorPart activeEditor = page.getActiveEditor();
		if (currentPart != null && currentPart != activeEditor) {
			deactivate();
			viewer = null;
			zoomManager = null;
			darstellung = null;
			currentPart = null;
		}

		if (activeEditor != null && currentPart != activeEditor) {
			final GraphicalViewer v = page.getActiveEditor().getAdapter(GraphicalViewer.class);
			final ZoomManager zm = page.getActiveEditor().getAdapter(ZoomManager.class);
			final Darstellung d = page.getActiveEditor().getAdapter(Darstellung.class);

			if (v != null && d != null && zm != null) {
				currentPart = part;
				darstellung = d;
				zoomManager = zm;
				viewer = v;
				activate();
			}
		}
	}

	/**
	 * Gibt den Graphical Viewer des aktuellen Workbench Parts zurück.
	 *
	 * @return der Viewer.
	 */
	protected GraphicalViewer getViewer() {
		return viewer;
	}

	/**
	 * Gibt den Zoom anager des aktuellen Workbench Parts zurück.
	 *
	 * @return der Zoom Manager.
	 */
	protected ZoomManager getZoomManager() {
		return zoomManager;
	}

	/**
	 * Gibt die Darstellung des aktuellen Workbench Parts zurück.
	 *
	 * @return der Darstellung.
	 */
	protected Darstellung getDarstellung() {
		return darstellung;
	}

	/**
	 * Wird aufgerufen, wenn der aktuelle Workbench Part für dieses Status Item
	 * verwendet werden kann.
	 *
	 * <p>
	 * Hier sollten Listener angemeldet werden.
	 */
	protected abstract void activate();

	/**
	 * Wird aufgerufen, wenn der aktuelle Workbench Part für dieses Status Item
	 * <em>nicht</em> verwendet werden kann.
	 *
	 * <p>
	 * Hier sollten Listener wieder abgemeldet werden.
	 */
	protected abstract void deactivate();

	@Override
	public void dispose() {
		page.removePartListener(listener);
		super.dispose();
	}
}
