/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Verschiebt ein Darstellungsobjekt von einer Darstellungsspalte auf eine
 * andere.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AddDoModelToDarstellungsSpalteCommand extends Command {

	private final DoModel model;
	private final DarstellungsSpalte darstellungsSpalte;
	private final Point location;
	private Point oldLocation;
	private DarstellungsSpalte oldDarstellungsSpalte;
	private int oldDarstellungsSpalteIndex;

	public AddDoModelToDarstellungsSpalteCommand(final DoModel model, final Point location,
			final DarstellungsSpalte darstellungsSpalte) {
		Assert.isNotNull(model, "model");
		Assert.isLegal(model.eContainer() instanceof Ebene, "model muss in einer Ebene liegen");
		Assert.isNotNull(location, "location");
		Assert.isNotNull(darstellungsSpalte, "darstellungsSpalte");

		this.model = model;
		this.location = location;
		this.darstellungsSpalte = darstellungsSpalte;

		setLabel("Objekt auf andere Darstellungsspalte verschieben");
	}

	@Override
	public void execute() {
		oldLocation = model.getLocation();
		oldDarstellungsSpalte = getOldDarstellungsSpalte();
		oldDarstellungsSpalteIndex = oldDarstellungsSpalte.getDoObjekte().indexOf(model);

		redo();
	}

	private DarstellungsSpalte getOldDarstellungsSpalte() {
		final Ebene ebene = (Ebene) model.eContainer();
		final StilisierteDarstellung darstellung = (StilisierteDarstellung) ebene.getDarstellung();
		for (final DarstellungsSpalte e : darstellung.getSpalten()) {
			if (e.getDoObjekte().contains(model)) {
				return e;
			}
		}

		// unreachable code
		throw new IllegalStateException("model muss in einer Darstellungsspalte liegen");
	}

	@Override
	public void redo() {
		model.setLocation(location);
		oldDarstellungsSpalte.getDoObjekte().remove(oldDarstellungsSpalteIndex);
		darstellungsSpalte.getDoObjekte().add(model);
	}

	@Override
	public void undo() {
		darstellungsSpalte.getDoObjekte().remove(model);
		oldDarstellungsSpalte.getDoObjekte().add(oldDarstellungsSpalteIndex, model);
		model.setLocation(oldLocation);
	}

}
