/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.commands;

import java.util.SortedSet;
import java.util.TreeSet;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.gef.commands.Command;

import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.model.DoModel;

/**
 * Löscht eine Darstellungsspalte in einer stilisierten Darstellung.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class DeleteDarstellungsSpalteCommand extends Command {

	private StilisierteDarstellung darstellung;
	private final DarstellungsSpalte spalte;
	private int spaltenIndexInDarstellung;
	private SortedSet<DoIdentifier> objekte;

	public DeleteDarstellungsSpalteCommand(final DarstellungsSpalte spalte) {
		Assert.isNotNull(spalte, "spalte");

		this.spalte = spalte;
	}

	@Override
	public void execute() {
		darstellung = getDarstellung();
		spaltenIndexInDarstellung = darstellung.getSpalten().indexOf(spalte);
		objekte = getObjekte();

		redo();
	}

	private StilisierteDarstellung getDarstellung() {
		final StilisierteDarstellung d = spalte.getDarstellung();
		Assert.isNotNull(d, "darstellung");
		return d;
	}

	private SortedSet<DoIdentifier> getObjekte() {
		final SortedSet<DoIdentifier> result = new TreeSet<>();
		for (final DoModel model : spalte.getDoObjekte()) {
			for (final Ebene ebene : darstellung.getEbenen()) {
				if (ebene.getDoObjekte().contains(model)) {
					final int index = ebene.getDoObjekte().indexOf(model);
					result.add(new DoIdentifier(model, ebene, index));
				}
			}
		}
		return result;
	}

	@Override
	public void redo() {
		removeDarstellungsSpalteFromDarstellung();
		if (!isLetzteSpalte()) {
			deleteObjekte();
			verkleinereDarstellungUmSpaltenBreite();
		}
	}

	private boolean isLetzteSpalte() {
		return darstellung.getSpalten().isEmpty();
	}

	private void deleteObjekte() {
		for (final Ebene ebene : darstellung.getEbenen()) {
			ebene.getDoObjekte().removeAll(spalte.getDoObjekte());
		}
	}

	private void removeDarstellungsSpalteFromDarstellung() {
		darstellung.getSpalten().remove(spaltenIndexInDarstellung);
	}

	private void verkleinereDarstellungUmSpaltenBreite() {
		final Dimension size = darstellung.getSize();
		darstellung.setSize(new Dimension(size.width - spalte.getWidth(), size.height));

	}

	@Override
	public void undo() {
		if (!isLetzteSpalte()) {
			vergroessereDarstellungUmSpaltenBreite();
			addObjekte();
		}
		addDarstellungsSpalteToDarstellung();
	}

	private void vergroessereDarstellungUmSpaltenBreite() {
		final Dimension size = darstellung.getSize();
		darstellung.setSize(new Dimension(size.width + spalte.getWidth(), size.height));
	}

	private void addObjekte() {
		for (final DoIdentifier id : objekte) {
			final DoModel model = id.model;
			final Ebene ebene = id.ebene;
			final int index = id.indexInEbene;
			ebene.getDoObjekte().add(index, model);
		}
	}

	private void addDarstellungsSpalteToDarstellung() {
		darstellung.getSpalten().add(spaltenIndexInDarstellung, spalte);
	}

	private static class DoIdentifier implements Comparable<DoIdentifier> {

		private final DoModel model;
		private final Ebene ebene;
		private final int indexInEbene;

		DoIdentifier(final DoModel model, final Ebene ebene, final int indexInEbene) {
			this.model = model;
			this.ebene = ebene;
			this.indexInEbene = indexInEbene;
		}

		@Override
		public int compareTo(final DoIdentifier o) {
			return Integer.compare(indexInEbene, o.indexInEbene);
		}

	}

}
