/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.dialogs;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.DialogCellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.progress.UIJob;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;
import com.bitctrl.lib.eclipse.emf.viewers.EmfLabelProvider;

import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.MassstaeblicheDarstellung;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.darstellung.util.AnsichtenEinstellungen;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungenEinstellungen;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.bitctrl.eclipse.viewers.EinstellungenContentProvider;

/**
 * Dialog zum Bearbeiten aller im System vorhanden Ansichten gleichzeitig. Dies
 * entspricht der <em>globalen Aktion</em> in den Anwenderforderungen. Für alle
 * vom Nutzer schreibbaren Ansichten, kann die Darstellung und der Ausschnitt
 * festgelegt werden. Die Ansichten mit ihrer Darstellung und ihrem Ausschnitt
 * werden tabellarisch präsentiert.
 *
 * @author BitCtrl Systems GmbH, Schumann
 *
 */
public class AnsichtenDialog extends TitleAreaDialog {

	private TreeViewer viewer;
	private final Set<ITreeSelection> changed = new HashSet<>();

	/**
	 * Initialisiert den Dialog
	 */
	public AnsichtenDialog(final Shell parentShell) {
		super(parentShell);
		setShellStyle(SWT.CLOSE | SWT.MIN | SWT.MAX | SWT.RESIZE);
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		setTitle("Ansichten");
		setMessage("In diesen Dialog können Sie global Aktionen für alle Ansichten ausführen.");

		final Composite container = (Composite) super.createDialogArea(parent);

		viewer = new TreeViewer(container, SWT.SINGLE | SWT.FULL_SELECTION);
		viewer.getControl().setLayoutData(GridDataFactory.fillDefaults().grab(true, true).create());
		viewer.setContentProvider(new EinstellungenContentProvider());
		viewer.getTree().setHeaderVisible(true);
		viewer.getTree().setLinesVisible(true);

		createAnsichtColumn();
		createDarstellungColumn();
		createAusschnittColumn();

		viewer.setInput(AnsichtenEinstellungen.INSTANCE);

		final Composite extraButtons = new Composite(container, SWT.NONE);
		GridDataFactory.fillDefaults().grab(true, false).applyTo(extraButtons);
		GridLayoutFactory.swtDefaults().applyTo(extraButtons);

		final Button removeAusschnittButton = new Button(extraButtons, SWT.PUSH);
		GridDataFactory.swtDefaults().applyTo(removeAusschnittButton);
		removeAusschnittButton.setText("Ausschnitt entfernen");
		removeAusschnittButton.setEnabled(false);
		removeAusschnittButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Object selected = ((IStructuredSelection) viewer.getSelection()).getFirstElement();
				if (selected instanceof Ansicht) {
					((Ansicht) selected).setAusschnitt(null);
					viewer.update(selected, null);
					changed.add((ITreeSelection) viewer.getSelection());
				}
			}
		});

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(final SelectionChangedEvent event) {
				final Object selected = ((IStructuredSelection) event.getSelection()).getFirstElement();
				if (selected instanceof Ansicht) {
					removeAusschnittButton.setEnabled(((Ansicht) selected).getAusschnitt() != null);
				} else {
					removeAusschnittButton.setEnabled(false);
				}
			}
		});

		return container;
	}

	private void createAnsichtColumn() {
		final TreeViewerColumn column = new TreeViewerColumn(viewer, SWT.NONE);
		column.getColumn().setText("Ansicht");
		column.getColumn().setWidth(200);

		column.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				if (element instanceof EObject) {
					return EmfUtil.getText((EObject) element);
				}

				return super.getText(element);
			}

		});
	}

	private void createDarstellungColumn() {
		final TreeViewerColumn column = new TreeViewerColumn(viewer, SWT.NONE);
		column.getColumn().setText("Darstellung");
		column.getColumn().setWidth(200);

		column.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				if (element instanceof Ansicht) {
					final Ansicht ansicht = (Ansicht) element;
					return EmfUtil.getText(ansicht.getDarstellung());
				}

				return "";
			}

		});

		column.setEditingSupport(new EditingSupport(viewer) {

			@Override
			protected void setValue(final Object element, final Object value) {
				if (value instanceof Darstellung) {
					final Ansicht ansicht = (Ansicht) element;
					final Darstellung prevDarst = ansicht.getDarstellung();
					final Darstellung newDarst = (Darstellung) value;

					if (checkReplacement(prevDarst, newDarst)) {
						ansicht.setDarstellung(newDarst);
					}
					getViewer().refresh(element);
				}
			}

			@Override
			protected Object getValue(final Object element) {
				final Ansicht ansicht = (Ansicht) element;
				return ansicht.getDarstellung();
			}

			@Override
			protected CellEditor getCellEditor(final Object element) {
				return new DialogCellEditor(viewer.getTree()) {

					@Override
					protected Object openDialogBox(final Control cellEditorWindow) {
						final ElementTreeSelectionDialog dlg = new ElementTreeSelectionDialog(getShell(),
								new EmfLabelProvider(), new EinstellungenContentProvider());
						dlg.setAllowMultiple(false);
						dlg.setValidator(new ISelectionStatusValidator() {

							@Override
							public IStatus validate(final Object[] selection) {
								for (final Object e : selection) {
									if (!(e instanceof Darstellung)) {
										return new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID,
												"Das ausgewählte Objekt ist keine Darstellung.");
									}
								}
								return Status.OK_STATUS;
							}

						});
						dlg.setInput(DarstellungenEinstellungen.INSTANCE);
						if (dlg.open() == Window.OK) {
							changed.add((ITreeSelection) getViewer().getSelection());
							return dlg.getFirstResult();
						}
						return null;
					}

					@Override
					protected void updateContents(final Object value) {
						if (getDefaultLabel() == null) {
							return;
						}

						getDefaultLabel().setText(EmfUtil.getText((EObject) value));
					}

				};
			}

			@Override
			protected boolean canEdit(final Object element) {
				return element instanceof Ansicht;
			}

		});
	}

	private void createAusschnittColumn() {
		final TreeViewerColumn column = new TreeViewerColumn(viewer, SWT.NONE);
		column.getColumn().setText("Ausschnitt");
		column.getColumn().setWidth(200);

		column.setLabelProvider(new ColumnLabelProvider() {

			@Override
			public String getText(final Object element) {
				if (element instanceof Ansicht) {
					final Ansicht ansicht = (Ansicht) element;
					return EmfUtil.getText(ansicht.getAusschnitt());
				}

				return "";
			}

		});
	}

	@Override
	protected void okPressed() {
		final Job job = new UIJob("Ansichten speichern") {

			@Override
			public IStatus runInUIThread(final IProgressMonitor monitor) {
				monitor.beginTask("Ansichten speichern", changed.size());

				for (final ITreeSelection s : changed) {
					final SpeicherKey art = (SpeicherKey) s.getPaths()[0].getFirstSegment();
					final Ansicht ansicht = (Ansicht) s.getPaths()[0].getLastSegment();
					try {
						AnsichtenEinstellungen.INSTANCE.setModellEinstellungen(art, ansicht.getName(), ansicht);

						// XXX Workaround, Vorgabe- und Sollparameter
						// überholen sich
						try {
							Thread.sleep(3000);
						} catch (final InterruptedException ex) {
							// kann ignoriert werden
						}

					} catch (final UrlasserDialogAbgebrochenException ex) {
						// Bediener hat den Urlasserdialog abgebrochen, das ist
						// ein
						// normaler
						// Vorgang
					} catch (final IOException ex) {
						ErrorDialog.openError(getShell(), "FEHLER", "Ansichten konnten nicht gespeichert werden!",
								new Status(IStatus.ERROR, DObjPlugin.PLUGIN_ID, ex.getLocalizedMessage(), ex));
					}
					monitor.worked(1);
				}

				monitor.done();

				return Status.OK_STATUS;
			}
		};
		job.setUser(true);
		job.schedule();

		super.okPressed();
	}

	private boolean checkReplacement(final Darstellung prevDarst, final Darstellung newDarst) {

		boolean result = true;

		if ((prevDarst instanceof MassstaeblicheDarstellung) && (newDarst instanceof StilisierteDarstellung)) {
			result = MessageDialog.openConfirm(getShell(), "Darstellung ersetzen",
					"Soll die maßstäbliche Darstellung tatsächlich durch eine stilisierte Darstellung ersetzt werden?");
		}

		if (result) {
			if ((prevDarst instanceof StilisierteDarstellung) && (newDarst instanceof MassstaeblicheDarstellung)) {
				result = MessageDialog.openConfirm(getShell(), "Darstellung ersetzen",
						"Soll die stilisierte Darstellung tatsächlich durch eine maßstäbliche Darstellung ersetzt werden?");
			}
		}

		if (result) {
			if ((prevDarst instanceof MassstaeblicheDarstellung) && (newDarst instanceof MassstaeblicheDarstellung)) {
				final double prevMinL = ((MassstaeblicheDarstellung) prevDarst).getMinimaleGeographischeLaenge();
				final double prevMinB = ((MassstaeblicheDarstellung) prevDarst).getMinimaleGeographischeBreite();
				final double prevMaxL = ((MassstaeblicheDarstellung) prevDarst).getMaximaleGeographischeLaenge();
				final double prevMaxB = ((MassstaeblicheDarstellung) prevDarst).getMaximaleGeographischeBreite();
				final double newMinL = ((MassstaeblicheDarstellung) newDarst).getMinimaleGeographischeLaenge();
				final double newMinB = ((MassstaeblicheDarstellung) newDarst).getMinimaleGeographischeBreite();
				final double newMaxL = ((MassstaeblicheDarstellung) newDarst).getMaximaleGeographischeLaenge();
				final double newMaxB = ((MassstaeblicheDarstellung) newDarst).getMaximaleGeographischeBreite();

				if ((prevMinL != newMinL) || (prevMaxL != newMaxL) || (prevMinB != newMinB) || (prevMaxB != newMaxB)) {

					result = MessageDialog.openConfirm(getShell(), "Darstellung ersetzen",
							"Koordinaten der Darstellungen stimmen nicht überein!\n\n"
									+ "Bisherige Koordinaten:\n  Minimale Länge: " + prevMinL + "\n  Minimale Breite: "
									+ prevMinB + "\n  Maxmiale Länge: " + prevMaxL + "\n  Maximale Breite: " + prevMaxB
									+ "\n" + "Neue Koordinaten:\n  Minimale Länge: " + newMinL + "\n  Minimale Breite: "
									+ newMinB + "\n  Maximale Länge: " + newMaxL + "\n  Maximale Breite: " + newMaxB
									+ "\n\nSoll die Darstellung trotzdem ersetzt werden?");
				}
			}
		}

		return result;
	}
}
