/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editors;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.SWTGraphics;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartFactory;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.nebula.paperclips.core.BigPrint;
import org.eclipse.nebula.paperclips.core.ImagePrint;
import org.eclipse.nebula.paperclips.core.grid.GridColumn;
import org.eclipse.nebula.paperclips.core.grid.GridPrint;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.UIJob;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.darstellung.actions.DarstellungActionConstants;
import de.bsvrz.buv.plugin.darstellung.actions.SelectAusschnittAction;
import de.bsvrz.buv.plugin.darstellung.actions.ToggleEbenenAction;
import de.bsvrz.buv.plugin.darstellung.actions.ToggleOverviewAction;
import de.bsvrz.buv.plugin.darstellung.actions.ToggleSynchronizeSelectionAction;
import de.bsvrz.buv.plugin.darstellung.editparts.DarstellungEditPartFactory;
import de.bsvrz.buv.plugin.darstellung.editparts.tree.DarstellungTreeEditPartFactory;
import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.util.AnsichtenEinstellungen;
import de.bsvrz.buv.plugin.dobj.actions.EllipseSelectionAction;
import de.bsvrz.buv.plugin.dobj.actions.FixViewportAction;
import de.bsvrz.buv.plugin.dobj.actions.PolygonSelectionAction;
import de.bsvrz.buv.plugin.dobj.actions.RectangleSelectionAction;
import de.bsvrz.buv.plugin.dobj.actions.StreckeSelectionAction;
import de.bsvrz.buv.plugin.dobj.actions.ToggleAntiKollisionsAlgorithmusAction;
import de.bsvrz.buv.plugin.dobj.actions.ToggleBaustelleGueltigAction;
import de.bsvrz.buv.plugin.dobj.actions.ToggleSelectionFeedbackAction;
import de.bsvrz.buv.plugin.dobj.actions.ToggleTouchedSelectionAction;
import de.bsvrz.buv.plugin.dobj.actions.ToggleVerbindungslinieAction;
import de.bsvrz.buv.plugin.dobj.editors.DobjGraphicalEditor;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.util.SelectionProperties;
import de.bsvrz.buv.rw.basislib.Rahmenwerk;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsEvent;
import de.bsvrz.buv.rw.basislib.dav.DefaultDavVerbindungsListener;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;
import de.bsvrz.buv.rw.basislib.legende.ITreeLegende;
import de.bsvrz.buv.rw.basislib.legende.LegendeWindow;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;

/**
 * Editor zur Anzeige einer Ansicht.
 *
 * <p>
 * Der Editor wird nicht zum Bearbeiten der Ansicht verwendet. Ein
 * Eclipse-Editor wird verwendet, damit die Ansicht immer im Zentrum des
 * Applikationsfensters dargestellt und über alle Perspektiven angezeigt wird.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class AnsichtEditor extends DobjGraphicalEditor<Ansicht> implements ITreeLegende, RwPrintable {

	/** die Kontext-Hilfe-ID. */
	private static final String HILFE_ID = DObjPlugin.PLUGIN_ID + "." + AnsichtEditor.class.getSimpleName();

	private final class InternalDavVerbindungsListener extends DefaultDavVerbindungsListener {
		@Override
		public void verbindungHergestellt(final DavVerbindungsEvent event) {
			final UIJob uiJob = new UIJob("Gehe online") {
				@Override
				public IStatus runInUIThread(final IProgressMonitor monitor) {
					geheOnline();
					return Status.OK_STATUS;
				}
			};
			uiJob.schedule();
		}

		@Override
		public void verbindungGetrennt(final DavVerbindungsEvent event) {
			final UIJob uiJob = new UIJob("Gehe offline") {
				@Override
				public IStatus runInUIThread(final IProgressMonitor monitor) {
					geheOffline();
					return Status.OK_STATUS;
				}
			};
			uiJob.schedule();
		}
	}

	private final class EbeneAdapter extends AdapterImpl {
		@Override
		public void notifyChanged(final Notification notification) {
			final Object notifier = notification.getNotifier();
			final int type = notification.getEventType();

			if (notifier instanceof Ebene) {
				final int featureID = notification.getFeatureID(Ebene.class);
				if (Notification.SET == type && DarstellungPackage.EBENE__VISIBLE == featureID) {
					final LegendeWindow legendeWindow = LegendeWindow.getCachedLegendeWindowFor(AnsichtEditor.this);
					if (legendeWindow != null) {
						legendeWindow.refresh();
					}
				}
			}
		}
	}

	/** Die ID des Editors. */
	public static final String EDITOR_ID = AnsichtEditor.class.getName();

	/**
	 * Initialisiert den Editor.
	 */
	public AnsichtEditor() {
		super(Ansicht.class);
	}

	private final Adapter ebeneAdapter = new EbeneAdapter();

	private SpeicherKey einstellungsArt;

	private InternalDavVerbindungsListener davVerbindungsListener;

	@Override
	public void createPartControl(final Composite parent) {
		super.createPartControl(parent);
		final Rahmenwerk rahmenWerk = RahmenwerkService.getService().getRahmenWerk();

		davVerbindungsListener = new InternalDavVerbindungsListener();
		rahmenWerk.addDavVerbindungsListener(davVerbindungsListener);
		if (rahmenWerk.isOnline()) {
			geheOnline();
		}
	}

	@Override
	public String getContributorId() {
		return DarstellungActionConstants.ANSICHT_PROPERTIES;
	}

	@Override
	protected EditPartFactory getEditPartFactory() {
		return new DarstellungEditPartFactory();
	}

	@Override
	protected void setInput(final IEditorInput input) {
		super.setInput(input);

		einstellungsArt = input.getAdapter(SpeicherKey.class);

		final Darstellung darstellung = getModel().getDarstellung();
		for (final Ebene ebene : darstellung.getEbenen()) {
			ebene.eAdapters().add(ebeneAdapter);
		}
	}

	@Override
	protected void configureGraphicalViewer() {
		super.configureGraphicalViewer();

		final ActionRegistry registry = getActionRegistry();

		// Allgemeine Actions

		// final IAction action0 = new ToggleDarstellungsLegendeAction(this,
		// getModel().getDarstellung());
		// registry.registerAction(action0);

		IAction action = new ToggleEbenenAction(this, getControl());
		registry.registerAction(action);

		action = new ToggleOverviewAction(this, getControl());
		registry.registerAction(action);

		action = new RectangleSelectionAction(this);
		registry.registerAction(action);

		action = new EllipseSelectionAction(this);
		registry.registerAction(action);

		action = new PolygonSelectionAction(this);
		registry.registerAction(action);

		action = new StreckeSelectionAction(this);
		registry.registerAction(action);

		action = new ToggleSynchronizeSelectionAction(this);
		registry.registerAction(action);

		action = new ToggleVerbindungslinieAction(this);
		registry.registerAction(action);

		action = new ToggleBaustelleGueltigAction(this);
		registry.registerAction(action);

		action = new FixViewportAction(this);
		registry.registerAction(action);

		// Actions die SelectionProperties benötigen

		final SelectionProperties selectionProperties = new SelectionProperties();
		selectionProperties.getMatchDoTypen().addAll(getModel().getDarstellung().getDoTypen());
		getGraphicalViewer().setProperty(SelectionProperties.class.toString(), selectionProperties);

		action = new ToggleTouchedSelectionAction(selectionProperties);
		registry.registerAction(action);

		action = new ToggleSelectionFeedbackAction(selectionProperties);
		registry.registerAction(action);

		action = new ToggleAntiKollisionsAlgorithmusAction(this);
		registry.registerAction(action);

		action = new SelectAusschnittAction(this);
		registry.registerAction(action);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(getControl(), AnsichtEditor.HILFE_ID);
	}

	@Override
	protected IContentOutlinePage getOutlinePage() {
		return new OutlinePage() {

			{
				setMaxBufferSize(100000);
			}

			@Override
			protected EditPartFactory getEditPartFactory() {
				return new DarstellungTreeEditPartFactory();
			}
		};
	}

	@Override
	public void doSave(final IProgressMonitor monitor) {
		final Ansicht ansicht = getModel();

		final IStatus status = EmfUtil.validate(ansicht, DObjPlugin.PLUGIN_ID);
		if (status.getSeverity() != IStatus.OK) {
			ErrorDialog.openError(getSite().getShell(), "Ansicht kann nicht gespeichert werden.",
					"Die zu speichernde Ansicht ist ungültig und kann nicht gespeichert werden.", status);
			DObjPlugin.getDefault().getLog().log(status);
			return;
		}

		try {
			AnsichtenEinstellungen.INSTANCE.setModellEinstellungen(einstellungsArt, ansicht.getName(), ansicht);
			getCommandStack().markSaveLocation();
		} catch (final UrlasserDialogAbgebrochenException ex) {
			// Bediener hat den Urlasserdialog abgebrochen, das ist ein normaler
			// Vorgang
		} catch (final IOException ex) {
			ErrorDialog.openError(getSite().getShell(), "FEHLER", "Ansicht konnte nicht gesichert werden!",
					new Status(IStatus.ERROR, DObjPlugin.PLUGIN_ID, ex.getLocalizedMessage(), ex));
		}
	}

	@Override
	public Control getControl() {
		return getGraphicalViewer().getControl().getParent();
	}

	@Override
	public List<ILegendeBaustein> getBausteine() {
		final List<ILegendeBaustein> bausteine = new ArrayList<>();

		bausteine.addAll(getBausteine(getGraphicalViewer().getRootEditPart()).values());

		return bausteine;
	}

	private Map<DoTyp, ILegendeBaustein> getBausteine(final EditPart editPart) {
		final SortedMap<DoTyp, ILegendeBaustein> bausteine = new TreeMap<>(new Comparator<DoTyp>() {

			@Override
			public int compare(final DoTyp dot1, final DoTyp dot2) {
				return dot1.getName().compareTo(dot2.getName());
			}

		});
		ILegendeBaustein baustein = editPart.getAdapter(ILegendeBaustein.class);
		if (baustein == null) {
			baustein = Platform.getAdapterManager().getAdapter(editPart, ILegendeBaustein.class);
		}

		// Nur sichtbare Ebenen berücksichtigen
		if (editPart.getModel() instanceof Ebene) {
			final Ebene ebene = (Ebene) editPart.getModel();
			if (!ebene.isVisible()) {
				return Collections.emptyMap();
			}
		}

		final Object model = editPart.getModel();
		if (model instanceof DoModel) {
			final DoTyp doTyp = ((DoModel) model).getDoTyp();
			if (baustein != null && !bausteine.containsKey(doTyp)) {
				bausteine.put(doTyp, baustein);
			}
		}

		for (final Object e : editPart.getChildren()) {
			bausteine.putAll(getBausteine((EditPart) e));
		}
		return bausteine;
	}

	@Override
	public Object getAdapter(@SuppressWarnings("rawtypes") final Class type) {
		if (type == Darstellung.class) {
			return getModel().getDarstellung();
		}

		return super.getAdapter(type);
	}

	@Override
	public PagePrint getDruckAuftrag() {

		final GridPrint grid = new GridPrint();
		grid.addColumn(new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));

		final Display d = getSite().getShell().getDisplay();

		final GraphicalEditPart rootEditPart = (GraphicalEditPart) getGraphicalViewer().getRootEditPart();

		final Rectangle swtR = new Rectangle(rootEditPart.getFigure().getBounds().x,
				rootEditPart.getFigure().getBounds().y, rootEditPart.getFigure().getBounds().width,
				rootEditPart.getFigure().getBounds().height);
		final Image image = new Image(d, swtR);
		final GC gc = new GC(image);
		final SWTGraphics swtg = new SWTGraphics(gc);
		rootEditPart.getFigure().paint(swtg);
		swtg.dispose();
		gc.dispose();
		grid.add(new ImagePrint(image.getImageData()));

		return new PagePrint(new BigPrint(grid));
	}

	@Override
	public String getTitel() {
		return getPartName();
	}

	@Override
	public void dispose() {

		if (davVerbindungsListener != null) {
			RahmenwerkService.getService().getRahmenWerk().removeDavVerbindungsListener(davVerbindungsListener);
		}

		final Darstellung darstellung = getModel().getDarstellung();
		for (final Ebene ebene : darstellung.getEbenen()) {
			ebene.eAdapters().remove(ebeneAdapter);
		}

		super.dispose();
	}

	@Override
	public Corner getDefaultCorner() {
		return Corner.TopLeft;
	}
}
