/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.darstellung.editparts;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.draw2d.LayoutManager;
import org.eclipse.draw2d.XYLayout;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.CompoundSnapToHelper;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.Request;
import org.eclipse.gef.SnapToGeometry;
import org.eclipse.gef.SnapToGrid;
import org.eclipse.gef.SnapToGuides;
import org.eclipse.gef.SnapToHelper;
import org.eclipse.gef.editpolicies.SnapFeedbackPolicy;
import org.eclipse.gef.rulers.RulerProvider;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;

import com.bitctrl.lib.eclipse.draw2d.svg.SVGCache;
import com.bitctrl.lib.eclipse.emf.util.EmfUtil;
import com.bitctrl.lib.eclipse.log.PluginLogger;

import de.bsvrz.buv.plugin.darstellung.editpolicies.EbeneEditPolicy;
import de.bsvrz.buv.plugin.darstellung.editpolicies.EbeneXYLayoutEditPolicy;
import de.bsvrz.buv.plugin.darstellung.figures.EbeneFigure;
import de.bsvrz.buv.plugin.darstellung.figures.SpaltenLayout;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.editparts.BaseGraphicalEditPart;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.requests.DobjRequestConstants;
import de.bsvrz.buv.plugin.dobj.requests.OnlineRequest;
import de.bsvrz.buv.plugin.dobj.util.BildManager;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.objekte.Bild;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttByte_JavaKeyword;

/**
 * Controller für eine Ebene der stilisierten Darstellung.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class EbeneEditPart extends BaseGraphicalEditPart<Ebene, EbeneFigure> {

	private final PluginLogger logger = DObjPlugin.getDefault().getLogger();
	private final boolean debug = DObjPlugin.getDefault().isDebugging();

	private boolean online;
	private Bild hintergrund;
	private ImageDescriptor hintergrundDesc;

	@Override
	protected List<?> getModelChildren() {
		if (getModel().getDarstellung() instanceof StilisierteDarstellung) {
			final StilisierteDarstellung darstellung = (StilisierteDarstellung) getModel().getDarstellung();
			if (!darstellung.getSpalten().isEmpty()) {
				return darstellung.getSpalten();
			}
		}

		return getModel().getDoObjekte();
	}

	@Override
	protected EbeneFigure createFigure() {
		return new EbeneFigure();
	}

	@Override
	public void activate() {
		super.activate();

		if (getModel().getDarstellung() instanceof StilisierteDarstellung) {
			((StilisierteDarstellung) getModel().getDarstellung()).eAdapters().add(this);
		}
	}

	@Override
	public void deactivate() {
		if (getParent().getModel() instanceof StilisierteDarstellung) {
			((StilisierteDarstellung) getParent().getModel()).eAdapters().remove(this);
		}

		super.deactivate();
	}

	@Override
	protected void createEditPolicies() {
		installEditPolicy(EditPolicy.COMPONENT_ROLE, new EbeneEditPolicy());
		installEditPolicy("Snap Feedback", new SnapFeedbackPolicy());
		installEditPolicy(EditPolicy.LAYOUT_ROLE, new EbeneXYLayoutEditPolicy());
	}

	@Override
	public void performRequest(final Request req) {
		super.performRequest(req);

		if (DobjRequestConstants.REQ_ONLINE.equals(req.getType())) {
			online = ((OnlineRequest) req).isOnline();

			if (debug) {
				logger.info(
						"Ebene " + EmfUtil.getText(getModel()) + " ist jetzt " + (online ? "online" : "offline") + ".");
			}
		}
	}

	@Override
	public void notifyChanged(final Notification notification) {
		final Object notifier = notification.getNotifier();
		final int type = notification.getEventType();

		if (notifier instanceof Ebene) {
			final int featureID = notification.getFeatureID(Ebene.class);
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DarstellungPackage.EBENE__VISIBLE:
					final Map<?, ?> editPartRegistry = getViewer().getEditPartRegistry();
					for (final DoModel doModel : getModel().getDoObjekte()) {
						final EditPart editPart = (EditPart) editPartRegistry.get(doModel);
						editPart.performRequest(new OnlineRequest(online && getModel().isVisible()));
					}
					//$FALL-THROUGH$
				case DarstellungPackage.EBENE__NAME:
				case DarstellungPackage.EBENE__EDITABLE:
				case DarstellungPackage.EBENE__HINTERGRUND:
				case DarstellungPackage.EBENE__HINTERGRUND_SKALIERUNG:
				case DarstellungPackage.EBENE__HINTERGRUND_LOCATION:
					refreshVisuals();
					break;
				default:
					break;
				}
				break;
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				switch (featureID) {
				case DarstellungPackage.EBENE__DO_OBJEKTE:
					refreshChildren();
					break;
				default:
					break;
				}
				break;
			default:
				break;
			}
		}

		if (notifier instanceof StilisierteDarstellung) {
			final int featureID = notification.getFeatureID(StilisierteDarstellung.class);
			switch (type) {
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				switch (featureID) {
				case DarstellungPackage.STILISIERTE_DARSTELLUNG__SPALTEN:
					refreshVisuals();
					refreshChildren();
					break;
				default:
					break;
				}
				break;
			default:
				break;
			}
		}
	}

	@Override
	protected void refreshVisuals() {
		final EbeneFigure f = getFigure();

		f.setVisible(getModel().isVisible());

		// Darstellungsspalten
		if (getModel().getDarstellung() instanceof StilisierteDarstellung) {
			final StilisierteDarstellung darstellung = (StilisierteDarstellung) getModel().getDarstellung();
			final LayoutManager layout = f.getLayoutManager();

			if (darstellung.getSpalten().isEmpty()) {
				if (!(layout instanceof XYLayout)) {
					f.setLayoutManager(new XYLayout());
				}
			} else {
				if (!(layout instanceof SpaltenLayout)) {
					f.setLayoutManager(new SpaltenLayout());
				}
				((SpaltenLayout) f.getLayoutManager()).setHeight(getModel().getDarstellung().getSize().height);
			}
		}

		// Hintergrundbild
		final Bild bild = getModel().getHintergrund();
		if (hintergrund == null || !hintergrund.equals(bild)) {
			hintergrund = bild;
			if (hintergrundDesc != null) {
				getResourceManager().destroyImage(hintergrundDesc);
			}

			if (bild != null) {
				if ("image/svg+xml".equalsIgnoreCase(bild.getKdBild().getDatum().getMIMEType())) {

					final Feld<AttByte_JavaKeyword> bilddaten = bild.getKdBild().getDatum().getBilddaten();

					final byte[] buffer = new byte[bilddaten.size()];
					for (int i = 0; i < bilddaten.size(); ++i) {
						buffer[i] = bilddaten.get(i).byteValue();
					}

					File file = null;
					FileOutputStream fout = null;
					try {
						file = DObjPlugin.getDefault().getBundle().getBundleContext()
								.getDataFile("hintergrund_svg" + System.currentTimeMillis() + ".tmp");
						fout = new FileOutputStream(file);
						fout.write(buffer);
						fout.flush();
						final SVGCache svg = new SVGCache();
						svg.setURI(file.toURI().toString(), true);
						f.setSvgCache(svg);
					} catch (final IOException ex) {
						Logger.getLogger(getClass().getName()).log(Level.SEVERE,
								"Laden eines SVG Hintergrundbildes ist fehlgeschlagen.", ex);
					} finally {
						if (fout != null) {
							try {
								fout.close();
							} catch (final IOException ex) {
								Logger.getLogger(getClass().getName()).log(Level.SEVERE,
										"Fehler beim Schliessen der temporären Datei!", ex);
							}
						}
						if (file != null) {
							file.delete();
						}
					}
				} else {
					hintergrundDesc = BildManager.INSTANCE.getImageDescriptor(bild);
					final Image image = getResourceManager().createImage(hintergrundDesc);
					f.setHintergrund(image);
				}
			} else {
				f.setHintergrund(null);
			}
		}
		f.setHintergrundSkalierung(getModel().getHintergrundSkalierung());
		f.setHintergrundLocation(getModel().getHintergrundLocation());
	}

	/**
	 * Wenn der Edit Part in einem Editor dargestellt wird, dann gibt die
	 * Methode folgende Adapter zurück.
	 *
	 * <ul>
	 * <li>{@link SnapToHelper}</li>
	 * </ul>
	 */
	@Override
	public Object getAdapter(final Class adapter) {
		if (adapter == SnapToHelper.class && isEditor()) {
			final List<SnapToHelper> snapStrategies = new ArrayList<>();
			Boolean val = (Boolean) getViewer().getProperty(RulerProvider.PROPERTY_RULER_VISIBILITY);
			if (val != null && val.booleanValue()) {
				snapStrategies.add(new SnapToGuides(this));
			}
			val = (Boolean) getViewer().getProperty(SnapToGeometry.PROPERTY_SNAP_ENABLED);
			if (val != null && val.booleanValue()) {
				snapStrategies.add(new SnapToGeometry(this));
			}
			val = (Boolean) getViewer().getProperty(SnapToGrid.PROPERTY_GRID_ENABLED);
			if (val != null && val.booleanValue()) {
				snapStrategies.add(new SnapToGrid(this));
			}

			if (snapStrategies.size() == 0) {
				return null;
			}
			if (snapStrategies.size() == 1) {
				return snapStrategies.get(0);
			}

			final SnapToHelper[] ss = new SnapToHelper[snapStrategies.size()];
			for (int i = 0; i < snapStrategies.size(); i++) {
				ss[i] = snapStrategies.get(i);
			}
			return new CompoundSnapToHelper(ss);
		}

		return super.getAdapter(adapter);
	}

}
