/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editparts.tree;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.DragTracker;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.Request;
import org.eclipse.gef.RequestConstants;
import org.eclipse.gef.tools.SelectEditPartTracker;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;

import de.bsvrz.buv.plugin.darstellung.editpolicies.EbeneEditPolicy;
import de.bsvrz.buv.plugin.darstellung.editpolicies.EbeneTreeEditPolicy;
import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.util.DarstellungIcons;
import de.bsvrz.buv.plugin.dobj.editparts.tree.BaseTreeEditPart;

/**
 * Tree Edit Part für eine {@link Ebene} (auch {@link AutoEbene}).
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class EbeneTreeEditPart extends BaseTreeEditPart<Ebene> {

	/*
	 * Diese drei Images dürfen nicht disposed werden. Sie werden vom
	 * ResourceManager des Viewers disposed.
	 */
	private Image ebeneVisible;
	private Image ebeneInvisible;
	private Image ebeneEditable;

	/**
	 * Gibt Darstellungsobjekte die in der Ebene angezeigt werden zurück.
	 *
	 * <p>
	 * Wenn die Darstellung das Root-Element ist, denn dann wird angenommen, das
	 * die Darstellung bearbeitet wird. Das ist z.&nbsp;B. im Vorpositionierer
	 * der Fall.
	 */
	@Override
	protected List<?> getModelChildren() {
		final List<Object> result = new ArrayList<>();

		// Objekte für die automatische Bestimmung der Objekte
		if (isEditor() && getModel() instanceof AutoEbene) {
			final AutoEbene autoEbene = (AutoEbene) getModel();
			result.addAll(autoEbene.getSystemObjectTypes());
			result.addAll(autoEbene.getConfigurationAreas());
			result.addAll(autoEbene.getMutableSets());
			result.addAll(autoEbene.getSystemObjects());
			result.addAll(autoEbene.getDoTypen());
		}

		// Angezeigte Darstellungsobjekte
		result.addAll(getModel().getDoObjekte());
		if (!isEditor() && getModel() instanceof AutoEbene) {
			result.addAll(((AutoEbene) getModel()).getAutoDoObjekte());
			Collections.sort(result, new Comparator<Object>() {

				@Override
				public int compare(final Object o1, final Object o2) {
					return String.CASE_INSENSITIVE_ORDER.compare(o1.toString(), o2.toString());
				}

			});
		}

		return result;
	}

	@Override
	protected Color getForeground() {
		if (!getModel().isVisible()) {
			return ColorConstants.gray;
		}
		return null;
	}

	@Override
	protected Image getImage() {
		// Icon für sichtbar
		if (ebeneEditable == null) {
			ebeneEditable = getViewer().getResourceManager()
					.createImage(DarstellungIcons.ActionEditEbene.getImageDescriptor());
		}
		if (getModel().isEditable()) {
			return ebeneEditable;
		}

		// Icon für nicht sichtbar
		if (ebeneVisible == null) {
			ebeneVisible = getViewer().getResourceManager()
					.createImage(DarstellungIcons.StatusEbeneVisible.getImageDescriptor());
			ebeneInvisible = getViewer().getResourceManager()
					.createImage(DarstellungIcons.StatusEbeneInvisible.getImageDescriptor());
		}
		if (getModel().isVisible()) {
			return ebeneVisible;
		}

		return ebeneInvisible;
	}

	@Override
	public void notifyChanged(final Notification notification) {
		final Object notifier = notification.getNotifier();
		final int type = notification.getEventType();
		final int featureID = notification.getFeatureID(Ebene.class);

		if (notifier instanceof Ebene) {
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DarstellungPackage.EBENE__NAME:
				case DarstellungPackage.EBENE__VISIBLE:
				case DarstellungPackage.EBENE__EDITABLE:
					refreshVisuals();
					break;
				}
				break;
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				switch (featureID) {
				case DarstellungPackage.EBENE__DO_OBJEKTE:
				case DarstellungPackage.AUTO_EBENE__SYSTEM_OBJECT_TYPES:
				case DarstellungPackage.AUTO_EBENE__CONFIGURATION_AREAS:
				case DarstellungPackage.AUTO_EBENE__MUTABLE_SETS:
				case DarstellungPackage.AUTO_EBENE__SYSTEM_OBJECTS:
				case DarstellungPackage.AUTO_EBENE__DO_TYPEN:
					refreshChildren();
					break;
				}
				break;
			}
		}
	}

	@Override
	protected void createEditPolicies() {
		super.createEditPolicies();

		installEditPolicy(EditPolicy.COMPONENT_ROLE, new EbeneEditPolicy());
		installEditPolicy(EditPolicy.PRIMARY_DRAG_ROLE, new EbeneTreeEditPolicy());
	}

	@Override
	public DragTracker getDragTracker(final Request req) {
		return new SelectEditPartTracker(this);
	}

	@Override
	public void performRequest(final Request req) {
		if (req.getType().equals(RequestConstants.REQ_OPEN)) {
			final SetCommand cmd = new SetCommand(getModel(), DarstellungPackage.Literals.EBENE__VISIBLE,
					!getModel().isVisible());
			getViewer().getEditDomain().getCommandStack().execute(cmd);
		}

		super.performRequest(req);
	}

}
