/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.editpolicies;

import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.editpolicies.LayoutEditPolicy;
import org.eclipse.gef.editpolicies.NonResizableEditPolicy;
import org.eclipse.gef.editpolicies.ResizableEditPolicy;
import org.eclipse.gef.editpolicies.XYLayoutEditPolicy;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.requests.CreateRequest;

import com.bitctrl.lib.eclipse.emf.gef.model.Sized;

import de.bsvrz.buv.plugin.darstellung.commands.AddDoModelToDarstellungsSpalteCommand;
import de.bsvrz.buv.plugin.darstellung.commands.CreateDoModelCommand;
import de.bsvrz.buv.plugin.darstellung.commands.MoveDoModelCommand;
import de.bsvrz.buv.plugin.darstellung.commands.ResizeDoModelCommand;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.dobj.HotspotConverter;
import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;

/**
 * {@link LayoutEditPolicy} für eine {@link DarstellungsSpalte}.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class DarstellungsSpalteXYLayoutEditPolicy extends XYLayoutEditPolicy {

	/**
	 * Erzeugt ein entsprechendes Command, wenn ein {@link DoModel} verschoben
	 * oder in der Größe geändert werden soll.
	 */
	@Override
	protected Command createChangeConstraintCommand(final ChangeBoundsRequest request, final EditPart child,
			final Object constraint) {
		if (child.getModel() instanceof DoModel && constraint instanceof Rectangle) {
			final DoModel model = (DoModel) child.getModel();
			final Rectangle bounds = (Rectangle) constraint;
			final Point location = getHotspotLocation(child, bounds);

			final CompoundCommand command = new CompoundCommand("Darstellungsobjekt Verschieben und Skalieren");
			command.add(new MoveDoModelCommand(model, location));

			if (model instanceof Sized) {
				command.add(new ResizeDoModelCommand(model, bounds.getSize()));
			}

			return command.unwrap();
		}

		// Fallback
		return null;
	}

	private Point getHotspotLocation(final EditPart child, final Rectangle bounds) {
		HotspotConverter hotspotConverter = child.getAdapter(HotspotConverter.class);
		if (hotspotConverter == null) {
			hotspotConverter = Platform.getAdapterManager().getAdapter(child, HotspotConverter.class);
		}

		Point location;
		if (hotspotConverter != null) {
			location = hotspotConverter.convertViewToModel(bounds.getLocation());
		} else {
			location = bounds.getLocation();
		}
		return location;
	}

	/**
	 * Erzeugt ein entsprechendes Command, wenn ein Objekt vom Typ {@link DoTyp}
	 * angelegt werden soll.
	 */
	@Override
	protected Command getCreateCommand(final CreateRequest request) {
		final Object childClass = request.getNewObjectType();
		if (childClass instanceof DoTyp) {
			final Ebene ebene = (Ebene) getHost().getParent().getModel();
			final DarstellungsSpalte spalte = (DarstellungsSpalte) getHost().getModel();
			final DoModel model = (DoModel) request.getNewObject();
			final Rectangle constraint = (Rectangle) getConstraintFor(request);
			final CreateDoModelCommand cmd = new CreateDoModelCommand(model, ebene, spalte);
			cmd.setLocation(constraint.getLocation());
			if (model instanceof Sized) {
				final Dimension size = constraint.getSize();
				if (!size.isEmpty()) {
					cmd.setSize(size);
				}
			}
			return cmd;
		}

		// Fallback
		return null;
	}

	@Override
	protected Command createAddCommand(final ChangeBoundsRequest request, final EditPart child,
			final Object constraint) {
		if (child.getModel() instanceof DoModel && constraint instanceof Rectangle) {
			final DarstellungsSpalte spalte = (DarstellungsSpalte) getHost().getModel();
			final DoModel model = (DoModel) child.getModel();
			final Rectangle bounds = (Rectangle) constraint;
			final Point location = getHotspotLocation(child, bounds);
			return new AddDoModelToDarstellungsSpalteCommand(model, location, spalte);
		}

		// fallback
		return super.createAddCommand(request, child, constraint);
	}

	/**
	 * Installiert für EditParts die ein {@link DoModel} als Modell haben
	 * zusätzlich eine {@link DoModelEditPolicy}.
	 *
	 * @return {@link NonResizableEditPolicy} bzw. {@link ResizableEditPolicy}
	 *         für {@link DoModel}-Edit Parts, sonst <code>null</code>.
	 */
	@Override
	protected EditPolicy createChildEditPolicy(final EditPart child) {
		if (child instanceof DoModelEditPart<?, ?>) {
			child.installEditPolicy(EditPolicy.COMPONENT_ROLE, new DoModelEditPolicy());

			if (child.getEditPolicy(PRIMARY_DRAG_ROLE) != null) {
				// Wenn das Kind schon eine PRIMARY_DRAG_ROLE - Editpolicy
				// hat, dann nehmen wir einfach die.
				return child.getEditPolicy(PRIMARY_DRAG_ROLE);
			}
			if (child.getModel() instanceof Sized) {
				return new ResizableEditPolicy();
			}
			return new NonResizableEditPolicy();
		}

		// fallback
		return null;
	}

}
