package de.bsvrz.buv.plugin.darstellung.model.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EContentAdapter;

import de.bsvrz.buv.plugin.darstellung.model.AutoEbene;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungPackage;
import de.bsvrz.buv.plugin.darstellung.model.DarstellungsSpalte;
import de.bsvrz.buv.plugin.darstellung.model.Ebene;
import de.bsvrz.buv.plugin.darstellung.model.StilisierteDarstellung;
import de.bsvrz.buv.plugin.dobj.model.ConfiguratedDoTyp;
import de.bsvrz.buv.plugin.dobj.model.DoKomponente;
import de.bsvrz.buv.plugin.dobj.model.DoKompositum;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.model.DobjPackage;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTypReferenz;

/**
 * Überwacht alle Objekte einer Darstellung auf Änderungen und stellt die
 * Konsistenz des Modells sicher.
 * 
 * <p>
 * Die folgenden Bedingungen müssen beachtet werden:
 * 
 * <ul>
 * <li>Für jedes {@link DoModel}s muss dessen {@link DoTyp} in der Darstellung
 * existieren.</li>
 * </ul>
 * 
 * @generated not
 * @author BitCtrl Systems GmbH, Falko Schumann
 * 
 */
public class DarstellungAdapter extends EContentAdapter {

	private boolean internalUpdate = false;

	@Override
	public boolean isAdapterForType(final Object type) {
		return type == Darstellung.class;
	}

	@Override
	public void notifyChanged(final Notification notification) {
		super.notifyChanged(notification);

		if (internalUpdate) {
			return;
		}

		if (notification.getNotifier() instanceof Darstellung) {
			handleDarstellung(notification);
		} else if (notification.getNotifier() instanceof Ebene) {
			handleEbene(notification);
		} else if (notification.getNotifier() instanceof DarstellungsSpalte) {
			handleDarstellungsSpalte(notification);
		} else if (notification.getNotifier() instanceof DoKompositum<?>) {
			handleDoKompositum(notification);
		}
	}

	private void handleDarstellung(final Notification notification) {
		if (notification.getFeature() == DarstellungPackage.eINSTANCE
				.getDarstellung_DoTypen()) {
			switch (notification.getEventType()) {
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.SET:
			case Notification.UNSET:
			case Notification.MOVE:
				sortDoTypenInDarstellung(notification);
				break;
			}
		}
	}

	private void sortDoTypenInDarstellung(final Notification notification) {
		final Darstellung darstellung = (Darstellung) notification
				.getNotifier();
		/*
		 * FIXME Sortierung wieder aktivieren. Die Sortierung funktioniert im
		 * Moment nicht, weil die DO-Typen am DoModel und der AutoEbene per
		 * Feldindex in Darstellung#doTypen referenziert werden. Der DoTyp muss
		 * erst eine ID bekommen und die muss zum Referenzieren auch verwendet
		 * werden. Dann kann das Sortieren wieder aktiviert werden.
		 */
		if (true) {
			return;
		}

		final DoTyp[] doTypen = (DoTyp[]) darstellung.getDoTypen().toArray();
		Arrays.sort(doTypen, new Comparator<DoTyp>() {

			@Override
			public int compare(final DoTyp doTyp1, final DoTyp doTyp2) {
				return doTyp1.getName().compareTo(doTyp2.getName());
			}

		});

		internalUpdate = true;
		darstellung.getDoTypen().clear();
		darstellung.getDoTypen().addAll(Arrays.asList(doTypen));
		internalUpdate = false;
	}

	private void handleDarstellungsSpalte(final Notification notification) {
		if (notification.getFeature() == DarstellungPackage.eINSTANCE
				.getDarstellungsSpalte_Width()) {
			switch (notification.getEventType()) {
			case Notification.SET:
			case Notification.UNSET:
				handleDarstellungsSpalteWidthChanged(notification);
				break;
			}
		}

	}

	private void handleDarstellungsSpalteWidthChanged(
			final Notification notification) {
		final DarstellungsSpalte spalte = (DarstellungsSpalte) notification
				.getNotifier();
		final StilisierteDarstellung darstellung = (StilisierteDarstellung) getDarstellung(spalte);
		int width = 0;
		for (final DarstellungsSpalte e : darstellung.getSpalten()) {
			width += e.getWidth();
		}
		final Dimension size = darstellung.getSize();
		darstellung.setSize(new Dimension(width, size.height));
	}

	private void handleDoKompositum(final Notification notification) {
		if (notification.getFeature() == DobjPackage.eINSTANCE
				.getDoKompositum_Komponenten()) {
			switch (notification.getEventType()) {
			case Notification.ADD:
			case Notification.ADD_MANY:
				handleAddDoModelToDoKompositum(notification);
				break;
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
				handleRemoveDoModelFromDoKompositum(notification);
				break;
			}
		}
	}

	private void handleAddDoModelToDoKompositum(final Notification notification) {
		@SuppressWarnings("unchecked")
		final DoKompositum<DoModel> kompositum = (DoKompositum<DoModel>) notification
				.getNotifier();
		final Darstellung darstellung = getDarstellung(kompositum);
		final Object value = notification.getNewValue();

		if (value instanceof List<?>) {
			@SuppressWarnings("unchecked")
			final List<DoModel> list = (List<DoModel>) value;
			for (final DoModel model : list) {
				addNonExistDoTypenToDarstellung(model, darstellung);
			}
		} else {
			addNonExistDoTypenToDarstellung((DoModel) value, darstellung);
		}
	}

	private static Darstellung getDarstellung(final EObject object) {
		final EObject container = object.eContainer();
		if (container instanceof Darstellung) {
			// Abbruchbedingung: Darstellung gefunden
			return (Darstellung) container;
		}

		Assert.isNotNull(container, "container");
		return getDarstellung(container);
	}

	private void handleRemoveDoModelFromDoKompositum(
			final Notification notification) {
		@SuppressWarnings("unchecked")
		final DoKompositum<DoModel> kompositum = (DoKompositum<DoModel>) notification
				.getNotifier();
		final Darstellung darstellung = getDarstellung(kompositum);
		final Object value = notification.getOldValue();

		if (value instanceof List<?>) {
			@SuppressWarnings("unchecked")
			final List<DoModel> list = (List<DoModel>) value;
			for (final DoModel model : list) {
				removeNonUsedDoTypenFromDarstellung(model, darstellung);
			}
		} else {
			removeNonUsedDoTypenFromDarstellung((DoModel) value, darstellung);
		}
	}

	private void handleEbene(final Notification notification) {
		if (notification.getFeature() == DarstellungPackage.eINSTANCE
				.getEbene_DoObjekte()) {
			switch (notification.getEventType()) {
			case Notification.ADD:
			case Notification.ADD_MANY:
				handleAddDoModelToEbene(notification);
				break;
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
				handleRemoveDoModelFromEbene(notification);
				break;
			}
		}
	}

	private void handleAddDoModelToEbene(final Notification notification) {
		final Ebene ebene = (Ebene) notification.getNotifier();
		final Darstellung darstellung = ebene.getDarstellung();
		final Object value = notification.getNewValue();

		if (value instanceof List<?>) {
			@SuppressWarnings("unchecked")
			final List<DoModel> list = (List<DoModel>) value;
			for (final DoModel model : list) {
				addNonExistDoTypenToDarstellung(model, darstellung);
			}
		} else {
			addNonExistDoTypenToDarstellung((DoModel) value, darstellung);
		}
	}

	private void addNonExistDoTypenToDarstellung(final DoModel model,
			final Darstellung darstellung) {
		final List<DoTyp> doTypen = getNonExistDoTypenAndUpdateModelWithExistDoTypen(
				darstellung, model);
		darstellung.getDoTypen().addAll(doTypen);
	}

	private void handleRemoveDoModelFromEbene(final Notification notification) {
		final Ebene ebene = (Ebene) notification.getNotifier();
		final Darstellung darstellung = ebene.getDarstellung();
		final Object value = notification.getOldValue();

		if (value instanceof List<?>) {
			@SuppressWarnings("unchecked")
			final List<DoModel> list = (List<DoModel>) value;
			for (final DoModel model : list) {
				removeNonUsedDoTypenFromDarstellung(model, darstellung);
			}
		} else {
			removeNonUsedDoTypenFromDarstellung((DoModel) value, darstellung);
		}
	}

	private void removeNonUsedDoTypenFromDarstellung(final DoModel model,
			final Darstellung darstellung) {
		final List<DoTyp> doTypen = getNonUsedDoTypen(darstellung, model);
		darstellung.getDoTypen().removeAll(doTypen);
	}

	/**
	 * Gibt alle Darstellungsobjekttypen eines Darstellungsobjekts zurück, die
	 * außer beim Darstellungsobjekt selbst nicht in einer Darstellung verwendet
	 * werden.
	 * 
	 * <p>
	 * Bei einfachen Darstellungsobjekten gibt es nur ein Darstellungsobjekttyp,
	 * bei zusammgesetzten können es aber mehrere sein.
	 * 
	 * <p>
	 * Die Methode ist hilfreich um beim Entfernen eines Darstellungsobjekts aus
	 * einer Darstellung die Darstellungsobjektypen zu bestimmen, die aus der
	 * Darstellung entfernt werden müssen.
	 */
	private static List<DoTyp> getNonUsedDoTypen(final Darstellung d,
			final DoModel model) {
		final List<DoTyp> result = new ArrayList<DoTyp>();
		final Set<DoTyp> usedDoTypen = getDoTypen(model);
		for (final DoTyp doTyp : usedDoTypen) {
			if (!isUsedDoTypInDarstellung(doTyp, d,
					Collections.singleton(model))) {
				result.add(doTyp);
			}
		}

		// fallback
		return result;
	}

	/**
	 * Gibt zu einem Darstellungsobjekt alle von ihm verwendeten
	 * Darstelungsobjekttypen zurück.
	 * 
	 * <p>
	 * Bei einfachen Darstellungsobjekten gibt es nur ein Darstellungsobjekttyp,
	 * bei zusammgesetzten können es aber mehrere sein.
	 */
	private static Set<DoTyp> getDoTypen(final DoModel model) {
		final Set<DoTyp> doTypen = new LinkedHashSet<DoTyp>();
		doTypen.add(model.getDoTyp());
		if (model instanceof DoKompositum<?>) {
			@SuppressWarnings("unchecked")
			final DoKompositum<DoModel> kompositum = (DoKompositum<DoModel>) model;
			for (final DoModel komponente : kompositum.getKomponenten()) {
				doTypen.addAll(getDoTypen(komponente));
			}
		}
		return doTypen;
	}

	/**
	 * Prüft ob es ein Darstellungsobjekt in einer Darstellung gibt, welches den
	 * angegebenen Darstellungsobjekttyp verwendet. Die Darstellungsobjekte in
	 * der Ausschlussliste werden dabei nicht mitgezählt.
	 */
	private static boolean isUsedDoTypInDarstellung(final DoTyp doTyp,
			final Darstellung darstellung, final Collection<DoModel> excludes) {
		for (final Ebene ebene : darstellung.getEbenen()) {
			if (ebene instanceof AutoEbene) {
				final AutoEbene autoEbene = (AutoEbene) ebene;
				for (final DoTyp dot : autoEbene.getDoTypen()) {
					if (dot == doTyp) {
						return true;
					}
				}
			}

			for (final DoModel model : ebene.getDoObjekte()) {
				if (excludes.contains(model)) {
					continue;
				} else if (model.getDoTyp() == doTyp) {
					return true;
				} else if (model instanceof DoKompositum<?>) {
					@SuppressWarnings("unchecked")
					final DoKompositum<DoModel> kompositum = (DoKompositum<DoModel>) model;
					if (isUsedDoTypInKompositum(doTyp, kompositum, excludes)) {
						return true;
					}
				}
			}
		}

		// fallback
		return false;
	}

	private static boolean isUsedDoTypInKompositum(final DoTyp doTyp,
			final DoKompositum<DoModel> kompositum,
			final Collection<DoModel> excludes) {
		for (final DoKomponente komponente : kompositum.getKomponenten()) {
			if (komponente instanceof DoModel) {
				final DoModel model = (DoModel) komponente;
				if (excludes.contains(model)) {
					continue;
				} else if (model.getDoTyp() == doTyp) {
					return true;
				} else if (model instanceof DoKompositum<?>) {
					@SuppressWarnings("unchecked")
					final DoKompositum<DoModel> k = (DoKompositum<DoModel>) model;
					if (isUsedDoTypInKompositum(doTyp, k, excludes)) {
						return true;
					}
				}
			}
		}

		// fallback
		return false;
	}

	/**
	 * Gibt die Darstellungsobjekttypen zurück, die von einem Darstellungsobjekt
	 * verwendet werden, aber sonst nicht in der Darstellung vorkommen.
	 * Darstellungsobjekttypen die das Darstellungsobjekt verwendet und bereits
	 * in der Darstellung bekannt sind, werden am Darstellungsobjekt durch die
	 * Darstellungsobjekttypen der Darstellung ersetzt.
	 * 
	 * <p>
	 * Bei einfachen Darstellungsobjekten gibt es nur ein Darstellungsobjekttyp,
	 * bei zusammgesetzten können es aber mehrere sein.
	 * 
	 * <p>
	 * Die Methode ist hilfreich um beim Hinzufügen eines Darstellungsobjekts zu
	 * einer Darstellung die Darstellungsobjektypen zu bestimmen, die der
	 * Darstellung hinzugefügt werden müssen. Zusätzlich wird das neue
	 * Darstellungsobjekt mit den bereits in der Darstellung vorhanden
	 * Darstellungsobjekten aktualisiert.
	 */
	private static List<DoTyp> getNonExistDoTypenAndUpdateModelWithExistDoTypen(
			final Darstellung d, final DoModel model) {
		return getDoTypAndUpdateModelIfNecessary(d, model,
				new ArrayList<DoTyp>());
	}

	private static List<DoTyp> getDoTypAndUpdateModelIfNecessary(
			final Darstellung d, final DoModel model, final List<DoTyp> doTypen) {
		// DO-Typ in Darstellung suchen
		DoTyp dot = getDoTyp(d.getDoTypen(), model.getDoTyp());
		// DO-Typ in als neu erkannten DO-Typen suchen
		if (dot == null) {
			dot = getDoTyp(doTypen, model.getDoTyp());
		}
		if (dot != null) {
			// DO-Typ ist bereits bekannt
			model.setDoTyp(dot);
		} else {
			// DO-Typ ist noch unbekannt
			doTypen.add(model.getDoTyp());
		}
		if (model instanceof DoKompositum<?>) {
			@SuppressWarnings("unchecked")
			final EList<DoModel> komponenten = ((DoKompositum<DoModel>) model)
					.getKomponenten();
			for (final DoModel komponente : komponenten) {
				getDoTypAndUpdateModelIfNecessary(d, komponente, doTypen);
			}
		}
		return doTypen;
	}

	/**
	 * Prüft, ob ein gegebener Darstellungsobjekttyp in einer Menge von
	 * Darstellungsobjekttypen enthalten ist.
	 * 
	 * @return ist der Darstellungsobjekttyp in der Menge enthalten, wird der
	 *         entsprechende Darstellungsobjekttyp <em>aus der Menge</em>
	 *         zurückgegeben. Sonst <code>null</code>.
	 */
	private static DoTyp getDoTyp(final Collection<DoTyp> doTypen,
			final DoTyp doTyp) {
		if (doTyp instanceof ConfiguratedDoTyp) {
			final ConfiguratedDoTyp modelDot = (ConfiguratedDoTyp) doTyp;
			for (final DoTyp dot : doTypen) {
				if (dot instanceof ConfiguratedDoTyp) {
					final ConfiguratedDoTyp darstellungDot = ((ConfiguratedDoTyp) dot);
					if (modelDot.getId().equals(darstellungDot.getId())) {
						return darstellungDot;
					}
				}
			}
		} else if (doTyp instanceof EditorDoTypReferenz) {
			final EditorDoTypReferenz modelDot = (EditorDoTypReferenz) doTyp;
			for (final DoTyp dot : doTypen) {
				if (dot instanceof EditorDoTypReferenz) {
					final EditorDoTypReferenz darstellungDot = ((EditorDoTypReferenz) dot);
					if (modelDot.getSystemObjekt().equals(
							darstellungDot.getSystemObjekt())) {
						return darstellungDot;
					}
				}
			}
		}

		// fallback
		return null;
	}

}
