/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.util;

import org.eclipse.core.runtime.Assert;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionPoint;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.model.MassstaeblicheDarstellung;
import de.bsvrz.buv.plugin.dobj.util.Projektion;

/**
 * Projekt bei der die Längen- und Breitengrade als x- und y-Koordinaten direkt
 * verwendet werden. Bei der quadratischen Plattkarte wird um Zerrungen in der
 * Darstullung zu verringern, abhänghängig von der Breite ein Faktor zur
 * Korrektur der Länge eingesetzt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class QuadratischePlattkarteProjektion implements Projektion {

	/**
	 * WGS-84-Koordinaten im Datenkatalog besitzen eine Genauigkeit von sechs
	 * Nachkommastellen, deswegen werden die Längen- und Breitengrade mit
	 * 1.000.000 multipliziert.
	 */
	private static final int DEFAULT_SKALIERUNG = 1000000;

	private double minimaleGeographischeLaenge;
	private double maximaleGeographischeBreite;
	private double laengenKorrektur;

	/**
	 * Gibt den Faktor für die Korrektur der geografischen Breite, abhängig von
	 * der geografischen Länge zurück.
	 *
	 * @return den Faktor der Längenkorrektur.
	 */
	public double getLaengenKorrektur() {
		return laengenKorrektur;
	}

	public void setDarstellung(final Darstellung darstellung) {
		Assert.isLegal(darstellung instanceof MassstaeblicheDarstellung);

		final MassstaeblicheDarstellung md = (MassstaeblicheDarstellung) darstellung;

		minimaleGeographischeLaenge = md.getMinimaleGeographischeLaenge();
		maximaleGeographischeBreite = md.getMaximaleGeographischeBreite();

		final double minimaleGeographischeBreite = md.getMinimaleGeographischeBreite();
		double breiteInRad = minimaleGeographischeBreite
				+ (maximaleGeographischeBreite - minimaleGeographischeBreite) / 2;
		breiteInRad = breiteInRad / 360 * 2 * Math.PI;
		laengenKorrektur = Math.cos(breiteInRad);
	}

	@Override
	public Point umrechnenModellNachWelt(final Point modell) {
		double dx = modell.preciseX();
		double dy = modell.preciseY();

		// Skalierung rückgängig machen
		dx /= DEFAULT_SKALIERUNG;
		dy /= DEFAULT_SKALIERUNG;

		// Längenkorrektur rückgängig machen
		dx /= getLaengenKorrektur();

		// Verschiebung nach (0,0) rückgängig machen
		dx += minimaleGeographischeLaenge;
		dy = maximaleGeographischeBreite - dy;

		return new PrecisionPoint(dx, dy);
	}

	@Override
	public Point umrechnenWeltNachModel(final Point welt) {
		double dx = welt.preciseX();
		double dy = welt.preciseY();

		// Verschieben der oberen linken Ecke der Karte nach (0,0)
		dx -= minimaleGeographischeLaenge;
		dy = maximaleGeographischeBreite - dy;

		// Einfache Längenkorrektur
		dx *= getLaengenKorrektur();

		// Skalierung der double-Werte in int-Werte
		dx *= DEFAULT_SKALIERUNG;
		dy *= DEFAULT_SKALIERUNG;

		return new PrecisionPoint(dx, dy);
	}

}
