/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.darstellung.views.ansicht;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.SWTGraphics;
import org.eclipse.gef.EditDomain;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.MouseWheelHandler;
import org.eclipse.gef.MouseWheelZoomHandler;
import org.eclipse.gef.commands.CommandStack;
import org.eclipse.gef.commands.CommandStackEvent;
import org.eclipse.gef.commands.CommandStackEventListener;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.ui.actions.ZoomInAction;
import org.eclipse.gef.ui.actions.ZoomOutAction;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.nebula.paperclips.core.BigPrint;
import org.eclipse.nebula.paperclips.core.ImagePrint;
import org.eclipse.nebula.paperclips.core.ScalePrint;
import org.eclipse.nebula.paperclips.core.grid.GridColumn;
import org.eclipse.nebula.paperclips.core.grid.GridPrint;
import org.eclipse.nebula.paperclips.core.page.PagePrint;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISaveablePart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.menus.CommandContributionItem;
import org.eclipse.ui.menus.CommandContributionItemParameter;
import org.eclipse.ui.part.ViewPart;

import de.bsvrz.buv.plugin.darstellung.actions.SelectAusschnittAction;
import de.bsvrz.buv.plugin.darstellung.actions.ToggleEbenenAction;
import de.bsvrz.buv.plugin.darstellung.actions.ToggleOverviewAction;
import de.bsvrz.buv.plugin.darstellung.actions.ToggleSynchronizeSelectionAction;
import de.bsvrz.buv.plugin.darstellung.model.Ansicht;
import de.bsvrz.buv.plugin.darstellung.model.Ausschnitt;
import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.darstellung.util.AnsichtenEinstellungen;
import de.bsvrz.buv.plugin.dobj.actions.ToggleVerbindungslinieAction;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.basislib.legende.ILegendeBaustein;
import de.bsvrz.buv.rw.basislib.legende.ITreeLegende;
import de.bsvrz.buv.rw.basislib.legende.LegendeWindow;
import de.bsvrz.buv.rw.basislib.printing.RwPrintable;

/**
 * {@link ViewPart} zur Anzeige von Darstellungen außerhalb von Eclipse
 * Editoren.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 * @author BitCtrl Systems GmbH, Christian Hösel
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 */
public class AnsichtsView extends ViewPart
		implements ITreeLegende, ISaveablePart, RwPrintable, CommandStackEventListener {

	/**
	 * Die View-ID dieses Views in der Eclipse Registry.
	 */
	public static final String VIEW_ID = AnsichtsView.class.getName();

	private AnsichtComposite ansichtComposite;
	private ViewPanAction panAction;
	private ViewRectangleZoomAction rectangleZoomAction;
	private ZoomInAction zoomInAction;
	private ZoomOutAction zoomOutAction;
	private SelectAusschnittAction selectAusschnittAction;

	private Ansicht ansicht;

	private ContributionItem legendeAction;
	private ToggleEbenenAction ebenenAction;
	private ToggleOverviewAction overViewAction;

	private ToggleSynchronizeSelectionAction toggleSynchronizeSelectionAction;

	private ToggleVerbindungslinieAction toggleVerbindungslinieAction;

	private Ausschnitt initialerAusschnitt;

	private String ausschnittName;

	@Override
	public void init(final IViewSite site) throws PartInitException {
		super.init(site);

		final String secondaryId = site.getSecondaryId();
		if (secondaryId == null) {
			throw new PartInitException("Es wurde kein Name für die anzuzeigende Ansicht angegeben");
		}

		final String[] strings = secondaryId.split("\\|");
		SpeicherKey speicherOrt = SpeicherKey.allgemeinNetzweit();
		String speicherOrtName = speicherOrt.getTypBezeichnung();
		String ansichtName = secondaryId;
		if (strings.length > 0) {
			speicherOrtName = strings[0];
			if (strings.length > 1) {
				ansichtName = strings[1];
			}
			if (strings.length > 2) {
				ausschnittName = strings[2];
			}
		}
		for (final SpeicherKey s : SpeicherKey.getDefaultKeys()) {
			if (s.getTypBezeichnung().equalsIgnoreCase(speicherOrtName)) {
				speicherOrt = s;
				break;
			}
		}

		if ((ansichtName != null) && (!ansichtName.isEmpty())) {
			ansicht = AnsichtenEinstellungen.INSTANCE.getModellEinstellungen(speicherOrt, ansichtName);
		}

		if (ansicht == null) {
			final IStatus status = new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID,
					"Die Ansicht \"" + ansichtName
							+ "\" existiert nicht. Die benötigte Ansicht muss vorher angelegt und in den "
							+ speicherOrt.getTypBezeichnung() + " Einstellungen unter dem Namen \"" + ansichtName
							+ "\" gespeichert werden.");
			throw new PartInitException(status);
		}

		initialerAusschnitt = ansicht.getAusschnitte().stream().filter(a -> a.getName().equals(ausschnittName))
				.findFirst().orElse(null);
		if (initialerAusschnitt == null && ausschnittName != null) {
			DObjPlugin.getDefault().getLog().log(new Status(IStatus.WARNING, DObjPlugin.PLUGIN_ID, "Der Ausschnitt \""
					+ ausschnittName + "\" konnte in der Ansicht \"" + ansichtName + "\" nicht gefunden werden."));
		}
	}

	@Override
	public void createPartControl(final Composite parent) {
		setPartName(ansicht.getName());
		ansichtComposite = new AnsichtComposite(parent, SWT.NONE, ansicht);
		makeActions();
		contributeToActionBars();

		final MenuManager menuManager = new MenuManager();
		final Menu menu = menuManager.createContextMenu(ansichtComposite);
		// Set the MenuManager
		ansichtComposite.getGraphicalViewer().getControl().setMenu(menu);
		ansichtComposite.getGraphicalViewer().setProperty(MouseWheelHandler.KeyGenerator.getKey(SWT.MOD1),
				MouseWheelZoomHandler.SINGLETON);
		getSite().registerContextMenu(menuManager, ansichtComposite.getGraphicalViewer());
		// Make the selection available
		getSite().setSelectionProvider(ansichtComposite.getGraphicalViewer());

		ansichtComposite.getEditDomain().getCommandStack().addCommandStackEventListener(this);

		if (initialerAusschnitt != null) {
			selectAusschnitt(initialerAusschnitt);
		}
	}

	@Override
	public void dispose() {
		if ((ansichtComposite != null) && (ansichtComposite.getEditDomain() != null)
				&& (ansichtComposite.getEditDomain().getCommandStack() != null)) {
			ansichtComposite.getEditDomain().getCommandStack().removeCommandStackEventListener(this);
		}
		super.dispose();
	}

	private void makeActions() {
		toggleSynchronizeSelectionAction = new ToggleSynchronizeSelectionAction(this);
		toggleVerbindungslinieAction = new ToggleVerbindungslinieAction(this);
		panAction = new ViewPanAction(this);
		ansichtComposite.getActionRegistry().registerAction(panAction);

		rectangleZoomAction = new ViewRectangleZoomAction(this);
		ansichtComposite.getActionRegistry().registerAction(rectangleZoomAction);

		final ZoomManager zoomManager = ansichtComposite.getZoomManager();

		selectAusschnittAction = new SelectAusschnittAction(this);
		ansichtComposite.getActionRegistry().registerAction(selectAusschnittAction);

		zoomInAction = new ZoomInAction(zoomManager);
		ansichtComposite.getActionRegistry().registerAction(zoomInAction);

		zoomOutAction = new ZoomOutAction(zoomManager);
		ansichtComposite.getActionRegistry().registerAction(zoomOutAction);

		final CommandContributionItemParameter parameter = new CommandContributionItemParameter(
				PlatformUI.getWorkbench(), null, LegendeWindow.COMMAND_ID_TOOGLE_LEGENDE_COMMAND_ID,
				CommandContributionItem.STYLE_CHECK);
		legendeAction = new CommandContributionItem(parameter);

		ebenenAction = new ToggleEbenenAction(this, getControl());
		ansichtComposite.getActionRegistry().registerAction(ebenenAction);

		overViewAction = new ToggleOverviewAction(this, getControl());
		ansichtComposite.getActionRegistry().registerAction(overViewAction);
	}

	private void contributeToActionBars() {
		final IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	private void fillLocalPullDown(final IMenuManager manager) {
		// tut nichts
	}

	private void fillLocalToolBar(final IToolBarManager manager) {
		manager.add(toggleSynchronizeSelectionAction);
		manager.add(toggleVerbindungslinieAction);
		manager.add(panAction);
		manager.add(rectangleZoomAction);
		manager.add(zoomInAction);
		manager.add(zoomOutAction);
		manager.add(legendeAction);
		manager.add(ebenenAction);
		manager.add(overViewAction);
		manager.add(selectAusschnittAction);
	}

	@Override
	public void setFocus() {
		if ((ansichtComposite != null) && (!ansichtComposite.isDisposed())) {
			ansichtComposite.setFocus();
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(@SuppressWarnings("rawtypes") final Class adapter) {
		if (EditDomain.class == adapter) {
			return ansichtComposite.getEditDomain();
		} else if (Darstellung.class == adapter) {
			return ansicht.getDarstellung();
		} else if (CommandStack.class == adapter) {
			return ansichtComposite.getEditDomain().getCommandStack();
		} else if (GraphicalViewer.class == adapter) {
			return ansichtComposite.getGraphicalViewer();
		} else if (ZoomManager.class == adapter) {
			return ansichtComposite.getGraphicalViewer().getProperty(ZoomManager.class.toString());
		}

		// fallback
		return super.getAdapter(adapter);
	}

	@Override
	public String getTitel() {
		return getPartName();
	}

	@Override
	public Corner getDefaultCorner() {
		return Corner.TopLeft;
	}

	@Override
	public List<ILegendeBaustein> getBausteine() {
		final List<ILegendeBaustein> bausteine = new ArrayList<>();

		bausteine.addAll(ansichtComposite.getBausteine().values());

		return bausteine;
	}

	@Override
	public Control getControl() {
		return ansichtComposite;
	}

	@Override
	public void doSave(final IProgressMonitor monitor) {
		monitor.beginTask("Speichere Ansicht...", IProgressMonitor.UNKNOWN);
		try {
			AnsichtenEinstellungen.INSTANCE.setModellEinstellungen(SpeicherKey.allgemeinNetzweit(), ansicht.getName(),
					ansicht);
			ansichtComposite.getEditDomain().getCommandStack().markSaveLocation();
			monitor.worked(1);
		} catch (final UrlasserDialogAbgebrochenException ex) {
			// durch den Nutzer abgebrochen.
			monitor.setCanceled(true);
		} catch (final IOException ex) {
			final String message = "Speichern einer Ansicht ist fehlgeschlagen.";
			MessageDialog.openError(getSite().getShell(), "Fehler", message);
			monitor.setCanceled(true);
		}
		monitor.done();
	}

	@Override
	public void doSaveAs() {
		//
	}

	@Override
	public boolean isDirty() {
		if (ansichtComposite == null) {
			return false;
		}
		return ansichtComposite.getEditDomain().getCommandStack().isDirty();
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public boolean isSaveOnCloseNeeded() {
		return true;
	}

	@Override
	public PagePrint getDruckAuftrag() {
		final GridPrint grid = new GridPrint();
		grid.addColumn(new GridColumn(GridColumn.DEFAULT_ALIGN, GridColumn.DEFAULT_SIZE, GridColumn.DEFAULT_WEIGHT));

		final Display d = getSite().getShell().getDisplay();

		final GraphicalEditPart rootEditPart = (GraphicalEditPart) ansichtComposite.getGraphicalViewer()
				.getRootEditPart();

		final Rectangle swtR = new Rectangle(rootEditPart.getFigure().getBounds().x,
				rootEditPart.getFigure().getBounds().y, rootEditPart.getFigure().getBounds().width,
				rootEditPart.getFigure().getBounds().height);
		final Image image = new Image(d, swtR);
		final GC gc = new GC(image);
		final SWTGraphics swtg = new SWTGraphics(gc);
		rootEditPart.getFigure().paint(swtg);
		swtg.dispose();
		gc.dispose();
		grid.add(new ImagePrint(image.getImageData()));

		if (MessageDialog.openQuestion(getViewSite().getShell(), "Ausdruck skalieren?",
				"Soll der Ausdruck auf genau 1 Seite skaliert werden?")) {
			return new PagePrint(new ScalePrint(new BigPrint(grid)));
		}
		return new PagePrint(new BigPrint(grid));

	}

	private void selectAusschnitt(final Ausschnitt ausschnitt) {
		final ZoomManager zoomManager = ansichtComposite.getZoomManager();
		zoomManager.setZoom(ausschnitt.getZoomLevel());
		zoomManager.getViewport().setHorizontalLocation(ausschnitt.getBounds().x);
		zoomManager.getViewport().setVerticalLocation(ausschnitt.getBounds().y);
	}

	@Override
	public void stackChanged(final CommandStackEvent event) {
		firePropertyChange(PROP_DIRTY);
		ansichtComposite.getEditDomain().getCommandStack();

	}

}
