/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.views.ansicht;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.Viewport;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.tools.AbstractTool;

import de.bsvrz.buv.plugin.dobj.util.DobjCursors;

/**
 * Mit diesem Tool kann in der Darstellung ein rechteckiger Bereich ausgewählt
 * werden, zu dem hineingezoomt wird.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @author BitCtrl Systems GmbH, Thomas Thierfelder
 */
class ViewRectangleZoomTool extends AbstractTool {

	public static final String PROP_ACTIVE = "active";

	private final PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);
	private ViewRectangleFeedbackFigure feedbackFigure;

	/**
	 * Initialisiert das Tool.
	 */
	ViewRectangleZoomTool() {
		setDefaultCursor(DobjCursors.ZoomIn.getCursor());
		setUnloadWhenFinished(false);
	}

	@Override
	protected String getCommandName() {
		return "zoom rectangle";
	}

	@Override
	protected void handleFinished() {
		/**
		 * TODO: Durch ZoomManager#zoomTo(Rectangle); ersetzen.
		 *
		 * Die Methode ist aber aktuell noch leer implementiert
		 */

		final Rectangle ausschnitt = new Rectangle(getStartLocation(), getLocation());
		if ((ausschnitt.getSize().height > 10) && (ausschnitt.getSize().width > 10)) {
			double currentZoom = getZoomManager().getZoom();

			final Point ursprung = new PrecisionPoint(
					(getViewport().getHorizontalRangeModel().getValue() / currentZoom)
							+ (ausschnitt.getLocation().preciseX() / currentZoom),
					(getViewport().getVerticalRangeModel().getValue() / currentZoom)
							+ (ausschnitt.getLocation().preciseY() / currentZoom));

			final double xZoom = currentZoom
					/ (ausschnitt.preciseWidth() / getViewport().getHorizontalRangeModel().getExtent());
			final double yZoom = currentZoom
					/ (ausschnitt.preciseHeight() / getViewport().getVerticalRangeModel().getExtent());

			if (Math.abs(currentZoom - xZoom) > Math.abs(currentZoom - yZoom)) {
				getZoomManager().setZoom(yZoom);
			} else {
				getZoomManager().setZoom(xZoom);
			}

			currentZoom = getZoomManager().getZoom();
			getViewport().setHorizontalLocation((int) Math.round(ursprung.preciseX() * currentZoom));
			getViewport().setVerticalLocation((int) Math.round(ursprung.preciseY() * currentZoom));
		}

		super.handleFinished();
	}

	protected ZoomManager getZoomManager() {
		return (ZoomManager) getCurrentViewer().getProperty(ZoomManager.class.toString());
	}

	private Viewport getViewport() {
		return getZoomManager().getViewport();
	}

	private IFigure getFeedbackFigure() {
		if (feedbackFigure == null) {
			feedbackFigure = new ViewRectangleFeedbackFigure();
			addFeedback(feedbackFigure);
		}
		return feedbackFigure;
	}

	protected void showFeedback() {
		final Rectangle rect = getSelectionRectangle().getCopy();
		getFeedbackFigure().translateToRelative(rect);
		getFeedbackFigure().setBounds(rect);
	}

	private Rectangle getSelectionRectangle() {
		return new Rectangle(getStartLocation(), getLocation());
	}

	protected void eraseFeedback() {
		if (feedbackFigure != null) {
			removeFeedback(feedbackFigure);
			feedbackFigure = null;
		}
	}

	@Override
	public void activate() {
		super.activate();
		propertyChangeSupport.firePropertyChange(PROP_ACTIVE, null, true);
	}

	@Override
	public void deactivate() {
		if (isInState(STATE_DRAG_IN_PROGRESS)) {
			eraseFeedback();
		}
		super.deactivate();
		setState(STATE_TERMINAL);
		propertyChangeSupport.firePropertyChange(PROP_ACTIVE, null, false);
	}

	@Override
	public boolean isActive() {
		return super.isActive();
	}

	@Override
	protected boolean handleDragInProgress() {
		if (isInState(STATE_DRAG | STATE_DRAG_IN_PROGRESS)) {
			showFeedback();
		}
		return true;
	}

	@Override
	protected boolean handleButtonDown(final int button) {
		if (button != 1) {
			setState(STATE_INVALID);
			handleInvalidInput();
		}
		stateTransition(STATE_INITIAL, STATE_DRAG_IN_PROGRESS);
		return true;
	}

	@Override
	protected boolean handleButtonUp(final int button) {
		if (stateTransition(STATE_DRAG_IN_PROGRESS, STATE_TERMINAL)) {
			eraseFeedback();
		}

		if (button == 1) {
			handleFinished();
		}
		return true;
	}

	@Override
	protected boolean handleInvalidInput() {
		eraseFeedback();
		return true;
	}

	public void addPropertyChangeListener(final PropertyChangeListener listener) {
		propertyChangeSupport.addPropertyChangeListener(listener);
	}

	public void removePropertyChangeListener(final PropertyChangeListener listener) {
		propertyChangeSupport.removePropertyChangeListener(listener);
	}

}
