/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.darstellung.wizards;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.PlatformUI;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungLocation;
import de.bsvrz.buv.rw.basislib.einstellungen.EinstellungOwnerType;
import de.bsvrz.buv.rw.basislib.einstellungen.SpeicherKey;
import de.bsvrz.buv.rw.basislib.einstellungen.UrlasserDialogAbgebrochenException;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenDialog;
import de.bsvrz.buv.rw.basislib.urlasser.UrlasserInfoDatenSender;
import de.bsvrz.buv.rw.bitctrl.eclipse.modell.emf.EMFModellEinstellungen;
import de.bsvrz.buv.rw.bitctrl.eclipse.wizards.EinstellungsArtWizardPage;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.puk.param.lib.daten.UrlasserInfo;

/**
 * Basisklasse für Wizards die {@link EObject}-Objekte importieren, die in den
 * Rahmenwerkseinstellungen gespeichert werden.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class AbstractImportWizard extends AbstractWorkbenchWizard {

	private final EMFModellEinstellungen einstellungen;

	private AbstractImportWizardPage importPage;
	private EinstellungsArtWizardPage einstellungsArtWizardPage;

	/**
	 * Initialisiert den Wizard.
	 *
	 * @param eClass
	 *            der Typ der zu importierenden Objekte.
	 */
	public AbstractImportWizard(final EClass eClass, final EMFModellEinstellungen<?> einstellungen) {
		super(eClass);
		this.einstellungen = einstellungen;
	}

	@Override
	public void addPages() {
		importPage = getImportWizardPage();
		addPage(importPage);

		einstellungsArtWizardPage = new EinstellungsArtWizardPage(EinstellungsArtWizardPage.class.getName());
		final Set<SpeicherKey> einstellungsArten = new HashSet<>();
		for (final Object e : (getSelection()).toList()) {
			if (e instanceof SpeicherKey) {
				einstellungsArten.add((SpeicherKey) e);
			}
		}
		einstellungsArtWizardPage.setPreSelection(einstellungsArten);
		addPage(einstellungsArtWizardPage);
	}

	protected abstract AbstractImportWizardPage getImportWizardPage();

	@Override
	public boolean performFinish() {
		final List<String> fileNames = importPage.getFileNames();
		for (final SpeicherKey einstellungsArt : einstellungsArtWizardPage.getEinstellungsArten()) {

			if ((einstellungsArt.getLocation() == EinstellungLocation.NETZWERKWEIT)
					&& (einstellungsArt.getOwnerType() == EinstellungOwnerType.SYSTEM)) {

				final UrlasserInfoDatenDialog dialog = new UrlasserInfoDatenDialog(getShell(),
						new UrlasserInfoDatenSender() {

							@Override
							public void execute(final ClientDavInterface verbindung, final UrlasserInfo urlasser) {
								for (final String fileName : fileNames) {
									final IStatus status = dateiImportieren(einstellungsArt, fileName, urlasser);
									if (status.matches(IStatus.ERROR)) {
										return;
									}
								}
							}
						});
				dialog.open();
			} else {
				for (final String fileName : fileNames) {
					final IStatus status = dateiImportieren(einstellungsArt, fileName, null);
					if (status.matches(IStatus.ERROR)) {
						return false;
					}
				}

			}
		}

		return true;
	}

	/**
	 * Einlesen einer Datei und speichern des Inhalts in den
	 * Benutzereinstellungen.
	 *
	 * @return Ergebnis des Imports
	 */
	private IStatus dateiImportieren(final SpeicherKey einstellungsArt, final String fileName,
			final UrlasserInfo urlasser) {
		final ResourceSet resSet = new ResourceSetImpl();
		final File file = new File(fileName);
		final Resource resource = resSet.createResource(URI.createURI(file.toURI().toString()));
		try {
			resource.load(null);
		} catch (final IOException ex) {
			final String message = "Die Datei konnte nicht gelesen werden: " + file;
			final IStatus status = DObjPlugin.getDefault().getLogger().error(message, ex);

			ErrorDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Fehler beim Import",
					message, status);

			return status;
		}

		// Dateiinhalt prüfen
		final EList<EObject> contents = resource.getContents();
		if (contents.size() != 1) {
			final String message = "Die Datei ist fehlerhaft und kann nicht gelesen werden: " + file;
			final IStatus status = DObjPlugin.getDefault().getLogger().error(message);

			ErrorDialog.openError(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), "Fehler beim Import",
					message, status);
			return status;
		}
		final Named named = (Named) contents.get(0);

		// Darstellung importieren
		try {
			if (einstellungen.getModellEinstellungen(einstellungsArt, named.getName()) != null) {
				if (MessageDialog.openQuestion(getShell(), getEClass().getInstanceTypeName() + " überschreiben",
						"Es existiert bereits ein Objekt mit dem selben Namen " + named.getName()
								+ ". Wollen Sie es überschreiben?")) {
					einstellungen.setModellEinstellungen(einstellungsArt, named.getName(), named, urlasser);
				} else {
					return Status.CANCEL_STATUS;
				}
			} else {
				einstellungen.setModellEinstellungen(einstellungsArt, named.getName(), named, urlasser);
			}
		} catch (final UrlasserDialogAbgebrochenException ex) {
			// Bediener hat den Urlasserdialog abgebrochen
			return Status.CANCEL_STATUS;
		} catch (final IOException ex) {
			final Status status = new Status(IStatus.ERROR, DObjPlugin.PLUGIN_ID, ex.getLocalizedMessage(), ex);
			ErrorDialog.openError(importPage.getShell(), "FEHLER",
					"Der Import konnte nicht erfolgreich ausgeführt werden!", status);
			return status;
		}
		return Status.OK_STATUS;
	}

}
