/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.gef.GraphicalEditPart;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.ZoomStufe;

/**
 * Mediator, der die Standardeigenschaften eines Darstellungsobjekts vom Modell
 * in die Figure überträgt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class DoMediator {

	private final DoModelEditPart<?, ?> editPart;

	/**
	 * Initialisiert den Mediator mit dem Edit Part, dessen Modell und Figure
	 * synchronisiert werden sollen.
	 */
	public DoMediator(final DoModelEditPart<?, ?> editPart) {
		this.editPart = editPart;
	}

	/**
	 * Überträgt die Standardeigenschaften vom Modell in die Figure.
	 */
	public void mediate() {
		mediateLocation();
		mediateZoomStufe();
		mediateZoom();
	}

	private void mediateLocation() {
		final DoModel model = editPart.getModel();
		final Point location = model.getAbsoluteLocation();
		if (location != null) {
			final DoFigure f = getFigure();
			f.setHotspot(location);

			final GraphicalEditPart parent = (GraphicalEditPart) editPart.getParent();
			parent.setLayoutConstraint(editPart, f, f.getBounds());
		}
	}

	private void mediateZoomStufe() {
		double vonZoom;
		double bisZoom;
		final DoModel model = editPart.getModel();
		final ZoomStufe zoomStufe = model.getDoTyp().getZoomStufe();
		if (zoomStufe != null) {
			final double zoomSkalierung = getZoomSkalierung();
			vonZoom = zoomStufe.getVonLevel();
			vonZoom /= 100;
			vonZoom /= zoomSkalierung;
			bisZoom = zoomStufe.getBisLevel();
			bisZoom /= 100;
			bisZoom /= zoomSkalierung;
		} else {
			vonZoom = 0.0;
			bisZoom = Double.MAX_VALUE;
		}
		getFigure().setSichtbareZoomStufe(vonZoom, bisZoom);
	}

	private double getZoomSkalierung() {
		final Darstellung darstellung = editPart.getDarstellung();
		if (darstellung != null) {
			return darstellung.getZoomSkalierung();
		}

		// fallback
		return 1.0;
	}

	private void mediateZoom() {
		final double zoom = editPart.getZoomManager().getZoom();
		getFigure().handleZoomChanged(zoom);
	}

	private DoFigure getFigure() {
		return (DoFigure) editPart.getFigure();
	}

}
