/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.actions;

import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.ui.IWorkbenchPart;

import de.bsvrz.buv.plugin.dobj.internal.DobjMessages;
import de.bsvrz.buv.plugin.dobj.util.DobjIcons;

/**
 * Aktiviert und Deaktiviert das Beibehalten des aktuellen Bildausschnitts bei
 * Veraenderung der Fenstergroesse.
 *
 * @author BitCtrl Systems GmbH, Thierfelder
 *
 */
public class FixViewportAction extends DobjWorkbenchPartAction {

	private final ControlListener controlListener = new ControlAdapter() {

		@Override
		public void controlResized(final ControlEvent e) {
			final ZoomManager zm = getZoomManager();
			final double newX = zm.getViewport().getHorizontalRangeModel().getExtent();
			final double newY = zm.getViewport().getVerticalRangeModel().getExtent();

			if (aktiv) {
				final double distX = newX - currentX;
				final double distY = newY - currentY;
				if (currentX > -1 && currentY > -1 && distX != 0 && distY != 0) {
					final double originX = zm.getViewport().getHorizontalRangeModel().getValue();
					final double originY = zm.getViewport().getVerticalRangeModel().getValue();
					final double currentZoom = zm.getZoom();
					double scale = 1.0;

					if (Math.abs(distX) >= Math.abs(distY)) {
						if (currentY != 0) {
							scale = newY / currentY;
						}
					} else {
						if (currentX != 0) {
							scale = newX / currentX;
						}
					}

					zm.setZoom(currentZoom * scale);
					zm.getViewport().getHorizontalRangeModel().setValue((int) Math.abs(originX * scale));
					zm.getViewport().getVerticalRangeModel().setValue((int) Math.abs(originY * scale));
				}
			}
			currentX = newX;
			currentY = newY;
		}

	};

	private boolean aktiv;
	private double currentX = -1.0;
	private double currentY = -1.0;

	/**
	 * Initialisiert die Action.
	 *
	 * @param part
	 *            der Workbench Part für den die Action gilt.
	 */
	public FixViewportAction(final IWorkbenchPart part) {
		super(part, AS_CHECK_BOX);

		setText(DobjMessages.ToggleFixViewport_Label);
		setImageDescriptor(DobjIcons.ActionToggleFixViewport.getImageDescriptor());
		setToolTipText(DobjMessages.ToggleFixViewport_Tooltip);

		setId(DobjActionFactory.TOGGLE_FIX_VIEWPORT.getCommandId());
		setActionDefinitionId(DobjActionFactory.TOGGLE_FIX_VIEWPORT.getCommandId());

		final ZoomManager zm = getZoomManager();
		if (zm != null) {
			currentX = zm.getViewport().getHorizontalRangeModel().getExtent();
			currentY = zm.getViewport().getVerticalRangeModel().getExtent();
		}
		getViewer().getControl().addControlListener(controlListener);
	}

	@Override
	protected boolean calculateEnabled() {
		return super.calculateEnabled() && getZoomManager() != null;
	}

	@Override
	public void run() {
		aktiv = isChecked();
	}

	@Override
	public void dispose() {
		if (getViewer().getControl() != null) {
			getViewer().getControl().removeControlListener(controlListener);
		}

		super.dispose();
	}

}
