/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.decorator;

import org.eclipse.draw2d.IFigure;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.resource.ResourceManager;

import de.bsvrz.buv.plugin.dobj.editparts.DoModelEditPart;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;

/**
 * Basisklasse für Vermittler zwischen den Dekorierermodell und der dekorierten
 * Figure.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <D>
 *            der Typ des Decorators.
 * @param <F>
 *            der Typ der Figure.
 */
public abstract class AbstractMediator<D extends EObject, F extends IFigure> {

	private final DoModelEditPart<?, ?> editPart;
	private final Class<D> decoratorType;
	private final EStructuralFeature feature;

	public AbstractMediator(final DoModelEditPart<?, ?> editPart, final Class<D> decoratorType,
			final EStructuralFeature feature) {
		this.editPart = editPart;
		this.decoratorType = decoratorType;
		this.feature = feature;
	}

	protected final DoModelEditPart<?, ?> getEditPart() {
		return editPart;
	}

	/**
	 * Liest den Wert aus dem Dekorierermodell und übergibt sie an die
	 * dekorierte Figure.
	 *
	 * <p>
	 * <em>Schablonenmethode:</em> Es wird falls vorhanden der Wert des
	 * {@link DoModel}s verwendet. Ist dieser nicht vorhanden oder nicht
	 * gesetzt, wird der Wert vom {@link DoTyp} an die Figure übergeben.
	 *
	 * @see #mediate(EObject)
	 */
	public final void mediate() {
		if (useModelValue()) {
			mediate(getModel());
		} else {
			mediate(getDoTyp());
		}
	}

	private boolean useModelValue() {
		return (isModelDecorated() && getModel().eIsSet(feature)) || !isDoTypDecorated();
	}

	/**
	 * Liest den Wert aus dem Dekorierermodell und übergibt sie an die
	 * dekorierte Figure.
	 *
	 * @see #mediate()
	 */
	protected abstract void mediate(final D decorator);

	private boolean isModelDecorated() {
		return decoratorType.isInstance(editPart.getModel());
	}

	protected final D getModel() {
		return (D) editPart.getModel();
	}

	private boolean isDoTypDecorated() {
		return decoratorType.isInstance(editPart.getModel().getDoTyp());
	}

	protected final D getDoTyp() {
		return (D) editPart.getModel().getDoTyp();
	}

	protected final F getFigure() {
		return (F) editPart.getFigure();
	}

	protected final ResourceManager getResourceManager() {
		return editPart.getViewer().getResourceManager();
	}

}
