/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editors;

import java.net.URL;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.gef.palette.CreationToolEntry;
import org.eclipse.gef.palette.MarqueeToolEntry;
import org.eclipse.gef.palette.PaletteContainer;
import org.eclipse.gef.palette.PaletteDrawer;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.palette.PaletteToolbar;
import org.eclipse.gef.palette.PanningSelectionToolEntry;
import org.eclipse.gef.palette.ToolEntry;
import org.eclipse.gef.requests.CreationFactory;
import org.eclipse.jface.resource.ImageDescriptor;

import com.bitctrl.lib.eclipse.emf.util.EmfUtil;

import de.bsvrz.buv.plugin.dobj.model.ConfiguratedDoTyp;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.util.DoTypenUtil;
import de.bsvrz.dav.daf.main.config.SystemObjectInfo;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.util.SystemObjectComparator;

public class DobjEditorPaletteFactory {

	private static class DoTypComparator implements Comparator<DoTyp> {

		@Override
		public int compare(final DoTyp o1, final DoTyp o2) {
			return Collator.getInstance().compare(o1.getName(), o2.getName());
		}

	}

	private static class DoTypCreationFactory implements CreationFactory {

		private final DoTyp doTyp;

		public DoTypCreationFactory(final DoTyp doTyp) {
			this.doTyp = doTyp;
		}

		@Override
		public Object getNewObject() {
			return doTyp.createModel();
		}

		@Override
		public Object getObjectType() {
			return doTyp;
		}

	}

	public DobjEditorPaletteFactory() {
		doTypen = new HashMap<>();
		doTypenOhneSystemObjekt = new ArrayList<>();
		for (final DoTyp doTyp : DoTypenUtil.getDoTypen()) {

			final SystemObjectType type = doTyp.getSystemObjectType();
			if (type != null) {
				if (!doTypen.containsKey(type)) {
					doTypen.put(type, new ArrayList<DoTyp>());
				}
				doTypen.get(type).add(doTyp);
			} else {
				doTypenOhneSystemObjekt.add(doTyp);
			}
		}

		systemObjectTypes = new ArrayList<>();
		systemObjectTypes.addAll(doTypen.keySet());

		// Inhalte Sortieren
		Collections.sort(systemObjectTypes, new SystemObjectComparator());
		Collections.sort(doTypenOhneSystemObjekt, new DoTypComparator());
		for (final List<DoTyp> e : doTypen.values()) {
			Collections.sort(e, new DoTypComparator());
		}
	}

	private final List<SystemObjectType> systemObjectTypes;
	private final Map<SystemObjectType, List<DoTyp>> doTypen;
	private final List<DoTyp> doTypenOhneSystemObjekt;

	public PaletteRoot createPalette() {
		final PaletteRoot palette = new PaletteRoot();

		palette.add(createToolsGroup(palette));
		palette.addAll(createDoTypenDrawers());

		return palette;
	}

	protected PaletteToolbar createToolsGroup(final PaletteRoot palette) {
		final PaletteToolbar toolbar = new PaletteToolbar("Tools");

		final ToolEntry tool = new PanningSelectionToolEntry();
		toolbar.add(tool);
		palette.setDefaultEntry(tool);

		toolbar.add(new MarqueeToolEntry());

		return toolbar;
	}

	protected List<PaletteContainer> createDoTypenDrawers() {
		final List<PaletteContainer> result = new ArrayList<>();
		for (final SystemObjectType systemObjectType : systemObjectTypes) {
			final PaletteContainer systemObjectTypeDrawer = createSystemObjectTypeDrawer(systemObjectType);
			if (!systemObjectTypeDrawer.getChildren().isEmpty()) {
				result.add(systemObjectTypeDrawer);
			}
		}

		result.add(createOtherDrawer());
		return result;
	}

	private PaletteContainer createSystemObjectTypeDrawer(final SystemObjectType systemObjectType) {
		final PaletteDrawer result = new PaletteDrawer(systemObjectType.getName());
		final SystemObjectInfo info = systemObjectType.getInfo();
		final String description = info.getShortInfo().replaceAll("\\s+", " ").trim();
		result.setDescription(description);
		result.setInitialState(PaletteDrawer.INITIAL_STATE_CLOSED);

		for (final DoTyp doTyp : doTypen.get(systemObjectType)) {
			if (doTyp instanceof ConfiguratedDoTyp && ((ConfiguratedDoTyp) doTyp).isNurMassstaeblicheDarstellung()) {
				continue;
			}

			ImageDescriptor icon = doTyp.getIcon();
			if (icon == null) {
				final IItemLabelProvider labelProvider = EmfUtil.getItemLabelProvider(doTyp);
				if (labelProvider != null) {
					final Object image = labelProvider.getImage(doTyp);
					if (image instanceof ImageDescriptor) {
						icon = (ImageDescriptor) image;
					} else if (image instanceof URL) {
						icon = ImageDescriptor.createFromURL((URL) image);
					}
				}
			}
			final String name = doTyp.getName();
			String tooltip = doTyp.getDescription();
			if (tooltip == null || tooltip.isEmpty()) {
				tooltip = name;
			}
			tooltip += " (" + doTyp.getId() + ")";
			final CreationToolEntry entry = new CreationToolEntry(name, tooltip, getCreationFactory(doTyp), icon, icon);
			result.add(entry);
		}

		return result;
	}

	private PaletteContainer createOtherDrawer() {
		final PaletteDrawer result = new PaletteDrawer("Andere");

		for (final DoTyp doTyp : doTypenOhneSystemObjekt) {
			final String name = doTyp.getName();
			final String tooltip = doTyp.getDescription();
			final ImageDescriptor icon = doTyp.getIcon();
			final CreationToolEntry entry = new CreationToolEntry(name, tooltip, getCreationFactory(doTyp), icon, icon);
			result.add(entry);
		}

		return result;
	}

	protected CreationFactory getCreationFactory(final DoTyp doTyp) {
		return new DoTypCreationFactory(doTyp);
	}

}
