/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editors;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.draw2d.PositionConstants;
import org.eclipse.gef.ContextMenuProvider;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.SnapToGeometry;
import org.eclipse.gef.SnapToGrid;
import org.eclipse.gef.dnd.TemplateTransferDragSourceListener;
import org.eclipse.gef.dnd.TemplateTransferDropTargetListener;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.ui.actions.ActionRegistry;
import org.eclipse.gef.ui.actions.AlignmentAction;
import org.eclipse.gef.ui.actions.MatchHeightAction;
import org.eclipse.gef.ui.actions.MatchWidthAction;
import org.eclipse.gef.ui.actions.ToggleRulerVisibilityAction;
import org.eclipse.gef.ui.actions.ToggleSnapToGeometryAction;
import org.eclipse.gef.ui.palette.PaletteViewer;
import org.eclipse.gef.ui.parts.ScrollingGraphicalViewer;
import org.eclipse.gef.ui.rulers.RulerComposite;
import org.eclipse.jface.action.IAction;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.ActionFactory;

import com.bitctrl.lib.eclipse.emf.eclipse.model.Named;

import de.bsvrz.buv.plugin.dobj.actions.CopyDobjAction;
import de.bsvrz.buv.plugin.dobj.actions.PasteDobjAction;

/**
 * Basisklasse für Editoren die zum Bearbeiten von Darstellungsobjekten und
 * Darstellungen verwendet werden. Die Benutzung der Outline-View und
 * Properties-View (Tabbed Properties) ist vorbereitet.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 *            der Typ der zu bearbeitenden Objekte.
 */
public abstract class DobjGraphicalEditorWithPalette<T extends Named> extends DobjGraphicalEditor<T> {

	private static final int PALETTE_SIZE = 200;

	private PaletteViewer paletteViewer;
	private RulerComposite rulerContainer;

	protected DobjGraphicalEditorWithPalette(final Class<T> modelClass) {
		super(modelClass);
	}

	@Override
	protected void createGraphicalViewer(final Composite parent) {
		rulerContainer = new RulerComposite(parent, SWT.NONE);
		super.createGraphicalViewer(rulerContainer);
		rulerContainer.setGraphicalViewer((ScrollingGraphicalViewer) getGraphicalViewer());
	}

	@Override
	protected void initializeGraphicalViewer() {
		super.initializeGraphicalViewer();

		getGraphicalViewer().addDropTargetListener(new TemplateTransferDropTargetListener(getGraphicalViewer()));

		final CopyDobjAction copy = (CopyDobjAction) getActionRegistry().getAction(ActionFactory.COPY.getId());
		getGraphicalViewer().addSelectionChangedListener(copy);

	}

	/**
	 * Called to configure the viewer before it receives its contents.
	 */
	protected void configurePaletteViewer() {
	}

	@Override
	protected ContextMenuProvider createContextMenuProvider() {
		return new DobjEditorContextMenuProvider(getGraphicalViewer(), getActionRegistry());
	}

	/**
	 * Creates the palette on the given composite.
	 *
	 * @param parent
	 *            the composite
	 */
	protected void createPaletteViewer(final Composite parent) {
		final PaletteViewer viewer = new PaletteViewer();
		setPaletteViewer(viewer);
		viewer.createControl(parent);
		configurePaletteViewer();
		hookPaletteViewer();
		initializePaletteViewer();
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(final Composite parent) {
		final Splitter splitter = new Splitter(parent, SWT.HORIZONTAL);
		createPaletteViewer(splitter);
		createGraphicalViewer(splitter);
		splitter.maintainSize(getPaletteViewer().getControl());
		splitter.setFixedSize(getInitialPaletteSize());
		splitter.addFixedSizeChangeListener(new PropertyChangeListener() {
			@Override
			public void propertyChange(final PropertyChangeEvent evt) {
				handlePaletteResized(((Splitter) evt.getSource()).getFixedSize());
			}
		});
	}

	/**
	 * Returns the PaletteRoot for the palette viewer.
	 *
	 * @return the palette root
	 */
	protected abstract PaletteRoot getPaletteRoot();

	/**
	 * Returns the initial palette size in pixels. Subclasses may override this
	 * method to return a persisted value.
	 *
	 * @see #handlePaletteResized(int)
	 * @return the initial size of the palette in pixels.
	 */
	protected int getInitialPaletteSize() {
		return PALETTE_SIZE;
	}

	/**
	 * Returns the PaletteViewer.
	 *
	 * @return the palette viewer
	 */
	protected PaletteViewer getPaletteViewer() {
		return paletteViewer;
	}

	/**
	 * Called whenever the user resizes the palette.
	 *
	 * @param newSize
	 *            the new size in pixels
	 */
	protected void handlePaletteResized(final int newSize) {
	}

	/**
	 * Called when the palette viewer is set. By default, the EditDomain is
	 * given the palette viewer.
	 */
	protected void hookPaletteViewer() {
		getEditDomain().setPaletteViewer(paletteViewer);
	}

	/**
	 * Called to populate the palette viewer.
	 */
	protected void initializePaletteViewer() {
		getPaletteViewer().addDragSourceListener(new TemplateTransferDragSourceListener(getPaletteViewer()));
	}

	/**
	 * Sets the palette viewer
	 *
	 * @param paletteViewer
	 *            the palette viewer
	 */
	protected void setPaletteViewer(final PaletteViewer paletteViewer) {
		this.paletteViewer = paletteViewer;
	}

	/**
	 * Sets the {@link #getPaletteRoot() palette root} of the edit domain
	 *
	 * @see org.eclipse.gef.ui.parts.GraphicalEditor#setEditDomain(org.eclipse.gef.DefaultEditDomain)
	 */
	@Override
	protected void setEditDomain(final DefaultEditDomain ed) {
		super.setEditDomain(ed);
		getEditDomain().setPaletteRoot(getPaletteRoot());
	}

	@Override
	protected void createActions() {
		super.createActions();

		final ActionRegistry registry = getActionRegistry();

		IAction action = new CopyDobjAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new PasteDobjAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this, PositionConstants.LEFT);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this, PositionConstants.RIGHT);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this, PositionConstants.TOP);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this, PositionConstants.BOTTOM);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this, PositionConstants.CENTER);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new AlignmentAction((IWorkbenchPart) this, PositionConstants.MIDDLE);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new MatchWidthAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());

		action = new MatchHeightAction(this);
		registry.registerAction(action);
		getSelectionActions().add(action.getId());
	}

	@Override
	protected void configureGraphicalViewer() {
		super.configureGraphicalViewer();

		// Hifstools
		final IAction showRulers = new ToggleRulerVisibilityAction(getGraphicalViewer());
		getActionRegistry().registerAction(showRulers);
		final IAction snapAction = new ToggleSnapToGeometryAction(getGraphicalViewer());
		getActionRegistry().registerAction(snapAction);
	}

	/**
	 * Legt die Properties für den {@link GraphicalViewer} fest.
	 *
	 * Wird von {@link #configureGraphicalViewer()} aufgerufen.
	 */
	@Override
	protected void loadProperties() {
		super.loadProperties();
		getGraphicalViewer().setProperty(SnapToGeometry.PROPERTY_SNAP_ENABLED, Boolean.TRUE);
		getGraphicalViewer().setProperty(SnapToGrid.PROPERTY_GRID_ENABLED, Boolean.FALSE);
		getGraphicalViewer().setProperty(SnapToGrid.PROPERTY_GRID_VISIBLE, Boolean.FALSE);
	}

}
