/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editparts;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.editparts.ZoomListener;
import org.eclipse.gef.editpolicies.NonResizableEditPolicy;
import org.eclipse.gef.editpolicies.SelectionEditPolicy;

import com.bitctrl.lib.eclipse.gef.editpolicies.OnlySelectionEditPolicy;
import com.bitctrl.util.Tupel;

import de.bsvrz.buv.plugin.darstellung.model.Darstellung;
import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.DoMediator;
import de.bsvrz.buv.plugin.dobj.editpolicies.MqAlsLinieSelectionEditPolicy;
import de.bsvrz.buv.plugin.dobj.figures.SystemObjectToolTipFigure;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.kollision.KollisionsManager;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.dobj.model.DobjPackage;
import de.bsvrz.buv.plugin.dobj.model.ZoomStufe;
import de.bsvrz.buv.plugin.dobj.requests.BaustelleGueltigRequest;
import de.bsvrz.buv.plugin.dobj.requests.DobjRequestConstants;
import de.bsvrz.buv.plugin.dobj.requests.KollisionsAlgorithmusRequest;
import de.bsvrz.buv.plugin.dobj.requests.OnlineRequest;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.plugin.dobj.util.IDavZeitStempelManager;
import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Basisklasse für Edit Parts, die {@link DoModel}-Objekte kontrollieren.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public abstract class DoModelEditPart<T extends DoModel, F extends IFigure> extends BaseGraphicalEditPart<T, F>
		implements ZoomListener {

	private static Set<Tupel<String, String>> cachedData = new HashSet<>();
	private static final Debug LOGGER = Debug.getLogger();

	private boolean antikollisionsalgorithmus = true;
	private boolean nurGueltigeBaustellenAnzeigen = true;

	private boolean online;
	private SystemObjectToolTipFigure toolTip;

	/**
	 * Gibt, falls vorhanden, den Zeitstempelmanager zurück.
	 *
	 * @return der Zeitstempelmanager oder <code>null</code>, wenn keiner
	 *         existiert.
	 */
	public IDavZeitStempelManager getDavZeitStempelManager() {
		return (IDavZeitStempelManager) getViewer().getProperty(IDavZeitStempelManager.class.toString());
	}

	/**
	 * Gibt, falls vorhanden, den Kollisionsmanager zurück.
	 *
	 * @return der Kollisionsmanager oder <code>null</code>, wenn keiner
	 *         existiert.
	 */
	public KollisionsManager getKollisionsManager() {
		return (KollisionsManager) getViewer().getProperty(KollisionsManager.class.toString());
	}

	/**
	 * Flag, ob der Antikollisionsalgorithmus aktiviert ist.
	 */
	public boolean isAntikollisionsalgorithmus() {
		return antikollisionsalgorithmus;
	}

	/**
	 * Flag, ob nur gültige Baustellen angezeigt werden sollen.
	 */
	public boolean isNurGueltigeBaustellenAnzeigen() {
		return nurGueltigeBaustellenAnzeigen;
	}

	/**
	 * Gibt, falls vorhanden eine Projektion für die Darstellung zurück.
	 *
	 * @return die Projektion oder <code>null</code>, wenn es keine gibt.
	 */
	public Projektion getProjektion() {
		if (getViewer() == null) {
			return Projektion.NULL;
		}

		return (Projektion) getViewer().getProperty(Projektion.class.toString());
	}

	/**
	 * Installiert grundlegende Edit Policies.
	 *
	 * <p>
	 * Diese Methode darf in abgeleiteten Klassen nur erweitert werden.
	 *
	 * <em>Selektierbarkeit von Objekten:</em>
	 *
	 * <p>
	 * Zum Markieren des Objekts wird unter der Rolle
	 * {@link EditPolicy#LAYOUT_ROLE} eine Edit Policy registriert. Abhängig
	 * davon, ob der Edit Part in einem Editor dargestellt wird oder nicht muss
	 * eine entsprechende {@link SelectionEditPolicy} installiert sein.
	 *
	 * <p>
	 * Wird der Edit Part in einem Editor verwendet, soll die Selektion auf
	 * Move, Resize und andere Request reagieren können. Wird der Edit Part
	 * nicht in einem Editor verwendet, sollte er sich nur markieren lassen.
	 *
	 * <p>
	 * Die Defaultimplementierung installiert eine
	 * {@link NonResizableEditPolicy} im Editor und sonst eine
	 * {@link OnlySelectionEditPolicy}. In abgeleiteten Methoden genügt es in
	 * der Regel bei Bedarf folgenden Code Block einzufügen.
	 *
	 * <code><pre>
	 *   if (isEditor()) {
	 * 	   return new CustomSelectionEditPolicy();
	 * 	 }
	 * </pre></code>
	 *
	 * Damit wird nur der Selektionsmechanismus im Editor ersetzt und der
	 * Standardselektionmechanismus in Nicht-Editoren bleibt erhalten.
	 *
	 * @todo ggf. Role auf {@link EditPolicy#PRIMARY_DRAG_ROLE} ändern (!) oder
	 *       Möglichkeit zum Umschalten zwischen Gesamtselektion und Knoten
	 *       bearbeite SelectionTool scheint das zu unterstützen.
	 */
	@Override
	protected void createEditPolicies() {
		if (!isEditor()) {
			// installEditPolicy(EditPolicy.LAYOUT_ROLE,
			// new OnlySelectionEditPolicy());
			installEditPolicy(EditPolicy.LAYOUT_ROLE, new MqAlsLinieSelectionEditPolicy());
		}
	}

	@Override
	public void activate() {
		super.activate();

		getModel().getDoTyp().eAdapters().add(this);

		if (getFigure().getToolTip() == null) {
			getFigure().setToolTip(getToolTip());
		}

		if (getModel().getRelativZu() != null) {
			getModel().getRelativZu().eAdapters().add(this);
		}

		getZoomManager().addZoomListener(this);

		if (!(getFigure() instanceof DoFigure)) {
			refreshZoom();
		}
	}

	@Override
	public void deactivate() {
		performRequest(new OnlineRequest(false));

		getZoomManager().removeZoomListener(this);

		if (getModel().getRelativZu() != null) {
			getModel().getRelativZu().eAdapters().remove(this);
		}

		getModel().getDoTyp().eAdapters().remove(this);

		super.deactivate();
	}

	@Override
	public void performRequest(final Request req) {
		super.performRequest(req);

		if (DobjRequestConstants.REQ_ONLINE.equals(req.getType())) {
			performOnlineRequest((OnlineRequest) req);
		} else if (DobjRequestConstants.REQ_KOLLISIONSALGORITHMUS.equals(req.getType())) {
			performKollisionsAlgorithmusRequest((KollisionsAlgorithmusRequest) req);
		} else if (DobjRequestConstants.REQ_BAUSTELLE_GUELTIG.equals(req.getType())) {
			performBaustelleGueltigRequest((BaustelleGueltigRequest) req);
		}
	}

	private void performOnlineRequest(final OnlineRequest req) {
		if (req.isOnline()) {
			if (!online) {
				geheOnline();
				online = true;
			}
		} else {
			if (online) {
				geheOffline();
				online = false;
			}
		}
	}

	private void performKollisionsAlgorithmusRequest(final KollisionsAlgorithmusRequest kar) {
		antikollisionsalgorithmus = kar.isKollisionsAlgorithmusAn();
		refreshVisuals();
	}

	private void performBaustelleGueltigRequest(final BaustelleGueltigRequest bstg) {
		nurGueltigeBaustellenAnzeigen = bstg.isAlleGueltigenBaustellenAnzeigen();
		refreshVisuals();
	}

	/**
	 * Flag, ob der Edit Part online geschaltet ist.
	 *
	 * @return <code>true</code>, wenn der Edit Part online ist, sonst
	 *         <code>false</code>.
	 * @see #geheOnline()
	 * @see #geheOffline()
	 */
	protected boolean isOnline() {
		return online;
	}

	/**
	 * Wird aufgerufen, wenn sich das Darstellungsobjekt am Datenverteiler auf
	 * Onlinedaten anmelden soll.
	 *
	 * <p>
	 * Darf in abgeleitet Klassen nur erweitert werden.
	 *
	 * @see #isOnline()
	 */
	protected void geheOnline() {
		// tut nix
	}

	/**
	 * Wird aufgerufen, wenn sich das Darstellungsobjekt am Datenverteiler von
	 * Onlinedaten abmelden soll.
	 *
	 * <p>
	 * Darf in abgeleitet Klassen nur erweitert werden.
	 *
	 * @see #isOnline()
	 */
	protected void geheOffline() {
		// tut nix
	}

	@Override
	public void notifyChanged(final Notification notification) {
		final Object notifier = notification.getNotifier();
		final int type = notification.getEventType();

		if (notifier instanceof DoModel) {
			final int featureID = notification.getFeatureID(DoModel.class);
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DobjPackage.DO_MODEL__RELATIV_ZU:
					if (notification.getOldValue() == null && getModel().getRelativZu() != null) {
						getModel().getRelativZu().eAdapters().add(this);
					} else if (notification.getOldValue() != null && getModel().getRelativZu() == null) {
						((DoModel) notification.getOldValue()).eAdapters().remove(this);
					}
					break;
				case DobjPackage.DO_MODEL__LOCATION:
				case DobjPackage.DO_MODEL__NAME:
				case DobjPackage.DO_MODEL__SYSTEM_OBJECT:
				case DobjPackage.DO_MODEL__ABSOLUTE_LOCATION:
					refreshVisuals();
					break;
				}
				break;
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
				switch (featureID) {
				case DobjPackage.DO_MODEL__AUSGEHENDE_VERBINDUNGEN:
					refreshSourceConnections();
					break;
				case DobjPackage.DO_MODEL__EINGEHENDE_VERBINDUNGEN:
					refreshTargetConnections();
					break;
				}
			}
		} else if (notifier instanceof DoTyp) {
			final int featureID = notification.getFeatureID(DoModel.class);
			switch (type) {
			case Notification.SET:
				switch (featureID) {
				case DobjPackage.DO_TYP__ZOOM_STUFE:
					refreshVisuals();
					break;
				}
				break;
			}
		}
	}

	@Override
	protected void refreshVisuals() {
		super.refreshVisuals();

		if (getFigure() instanceof DoFigure) {
			new DoMediator(this).mediate();
		} else {
			/*
			 * XXX Fallback für alle die noch nicht das neue Interface DoFigure
			 * verwenden.
			 */
			refreshLocation();
			refreshZoom();
		}

		refreshTooltip();
	}

	private void refreshLocation() {
		if (!(getFigure() instanceof DoFigure)) {
			final Point location = getModel().getAbsoluteLocation();
			if (location != null) {
				final IFigure f = getFigure();
				f.setLocation(location);
				updateLayoutConstraint();
			}
		}
	}

	private void updateLayoutConstraint() {
		final GraphicalEditPart parent = (GraphicalEditPart) getParent();
		final IFigure f = getFigure();
		parent.setLayoutConstraint(this, f, f.getBounds());
	}

	private void refreshZoom() {
		zoomChanged(getZoomManager().getZoom());
	}

	private void refreshTooltip() {
		if (toolTip != null) {
			toolTip.setSystemObject(getModel().getSystemObject());
			if (getModel().getName() != null && !getModel().getName().isEmpty()) {
				toolTip.setName(getModel().getName());
			}
		}
	}

	@Override
	public void zoomChanged(final double zoom) {
		if (getFigure() instanceof DoFigure) {
			((DoFigure) getFigure()).handleZoomChanged(zoom);
		} else {
			/*
			 * XXX Fallback für alle die noch nicht das neue Interface DoFigure
			 * verwenden.
			 */
			final ZoomStufe zoomStufe = getModel().getDoTyp().getZoomStufe();
			if (zoomStufe != null) {
				double min = zoomStufe.getVonLevel();
				min /= 100;
				double max = zoomStufe.getBisLevel();
				max /= 100;
				final double readableZoom = zoom * getZoomManager().getUIMultiplier();
				getFigure().setVisible(min <= readableZoom && readableZoom <= max);
			} else {
				getFigure().setVisible(true);
			}
		}
	}

	/**
	 * Gibt den Tooltip des Edit Parts zurück. Der Tooltip wird im
	 * {@link #activate()} gesetzt, wenn die Figure noch keinen Tooltip besitzt.
	 *
	 * <p>
	 * Die Defaultimplementierung gibt {@link SystemObjectToolTipFigure} zurück.
	 * Dieser Tooltip zeigt den Namen, PID u.&nbsp;a. statische Eigenschaften
	 * des Systemobjekts an.
	 *
	 * <p>
	 * Kann in abgeleiteten Klassen entsprechend überschrieben werden.
	 * Insbesondere sollte hier <code>null</code> zurückgegeben werden, wenn ein
	 * Tooltip per Edit Policy registriert wird.
	 *
	 * @return der Tooltip.
	 */
	protected IFigure getToolTip() {
		if (toolTip == null) {
			toolTip = new SystemObjectToolTipFigure(getModel().getSystemObject());
		}
		return toolTip;
	}

	/**
	 * Liest für alle Systemobjekte eines Typs die angegebenen konfigurierenden
	 * Daten. Durch dieses Vorladen, können später die Daten einzeln schneller
	 * abgerufen werden.
	 * <p>
	 * Für jede Kombination aus Systemobjekttyp und Attributgruppe, werden die
	 * Daten nur einmal gelesen.
	 *
	 * @param typePid
	 *            die PID des Systemobjekttyps.
	 * @param atgPid
	 *            die PID der konfigurierenden Daten.
	 */
	protected void vorladen(final String typePid, final String atgPid) {
		final Tupel<String, String> cached = new Tupel<>(typePid, atgPid);
		if (cachedData.contains(cached)) {
			return;
		}

		long t = System.currentTimeMillis();

		final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
		final DataModel dataModel = dav.getDataModel();
		SystemObjectType systemObjectType;
		AttributeGroup atg;

		systemObjectType = dataModel.getType(typePid);
		atg = dataModel.getAttributeGroup(atgPid);
		dataModel.getConfigurationData(systemObjectType.getObjects(), atg);

		cachedData.add(cached);

		t = System.currentTimeMillis() - t;
		LOGGER.fine(
				getClass().getSimpleName() + "#vorladen(" + typePid + ", " + atgPid + "): " + (t / 1000) + " Sekunden");
	}

	/**
	 * Nicht sichtbare Objekte sind nicht auswählbar.
	 */
	@Override
	public boolean isSelectable() {
		if ((getEditorType() & DobjUtil.PROP_EDITOR_TYPE_DOTYP) == DobjUtil.PROP_EDITOR_TYPE_DOTYP) {
			return false;
		}

		return getFigure().isShowing() && super.isSelectable();
	}

	/**
	 * Gibt die Darstellung zurück, in der sich das Darstellungsobjekt befindet,
	 * oder <code>null</code>, wenn sich das Darstellungsobjekt nicht in einer
	 * Darstellung befindet. Letzteres kann z.&nbsp;B. der Fall sein, wenn das
	 * Darstellungsobjekt als einzelnes Objekt angezeigt wird.
	 */
	public Darstellung getDarstellung() {
		return getDarstellung(this);
	}

	private Darstellung getDarstellung(final EditPart part) {
		final EditPart parent = part.getParent();
		if (parent == null) {
			return null;
		}

		final Object parentModel = parent.getModel();
		if (parentModel instanceof Darstellung) {
			return (Darstellung) parentModel;
		}

		return getDarstellung(parent);
	}

	@Override
	protected List<?> getModelSourceConnections() {
		return getModel().getAusgehendeVerbindungen();
	}

	@Override
	protected List<?> getModelTargetConnections() {
		return getModel().getEingehendeVerbindungen();
	}

}
