/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.editparts;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;

import de.bsvrz.buv.plugin.dobj.util.Projektion;
import de.bsvrz.sys.funclib.bitctrl.geolib.WGS84Koordinate;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdKomplexKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Flaeche;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Komplex;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.KomplexXY;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.util.geolib.WGS84Util;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Basisklasse für Edit Parts die als Controller fuer Systemobjekte die
 * <code>typ.komplex</code> erweitern.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 * @param <T>
 */
public abstract class KomplexEditPart<T extends Komplex, F extends IFigure> extends GeoRefererenzObjektEditPart<T, F> {

	private static final Debug LOGGER = Debug.getLogger();
	// TODO Punkt- und Komplexobjekte einbeziehen

	private List<PointList> flaechen;
	private List<PointList> linien;

	@Override
	protected void cleanModellKoordinaten() {
		flaechen = null;
	}

	/**
	 * Bestimmt die in dem Komplex-Objekt enthaltenen Flächen.
	 */
	protected List<PointList> getFlaechen() {
		if (flaechen == null) {
			final Projektion aktuelleProjektion = getProjektion();
			final KomplexXY komplex = (KomplexXY) getSystemObjekt();
			try {
				flaechen = new ArrayList<>();
				final KdKomplexKoordinaten.Daten datum = komplex.getKdKomplexKoordinaten().getDatum();
				if (datum != null) {
					for (final Flaeche flaeche : datum.getFlaechenReferenz()) {
						final PointList points = new PointList();
						for (final WGS84Koordinate k : WGS84Util.konvertiere(flaeche)) {
							final Point p = aktuelleProjektion
									.umrechnenWeltNachModel(new PrecisionPoint(k.getLaenge(), k.getBreite()));
							points.addPoint(p);
						}
						flaechen.add(points);
					}
				}
			} catch (final RuntimeException ex) {
				LOGGER.error("Die WGS-84-Koordinaten des Komplex konnten nicht bestimmt werden: " + komplex, ex);
			}
		}
		return flaechen;
	}

	/**
	 * Bestimmt die in dem Komplex-Objekt enthaltenen Linien.
	 */
	protected List<PointList> getLinien() {
		if (linien == null) {
			final Projektion aktuelleProjektion = getProjektion();
			final KomplexXY komplex = (KomplexXY) getSystemObjekt();
			try {
				linien = new ArrayList<>();
				final KdKomplexKoordinaten.Daten datum = komplex.getKdKomplexKoordinaten().getDatum();
				if (datum != null) {
					for (final Linie linie : datum.getLinienReferenz()) {
						final PointList points = new PointList();
						for (final WGS84Koordinate k : WGS84Util.konvertiere(linie)) {
							final Point p = aktuelleProjektion
									.umrechnenWeltNachModel(new PrecisionPoint(k.getLaenge(), k.getBreite()));
							points.addPoint(p);
						}
						linien.add(points);
					}
				}
			} catch (final RuntimeException ex) {
				LOGGER.error("Die WGS-84-Koordinaten des Komplex konnten nicht bestimmt werden: " + komplex, ex);
			}
		}
		return linien;
	}
}
