/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.figures;

import org.eclipse.draw2d.GridLayout;
import org.eclipse.draw2d.Label;
import org.eclipse.draw2d.Panel;

import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.preferences.DobjPreference;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Ein einfacher Tooltip, der für ein {@link SystemObject} den Namen und die PID
 * anzeigt. Zusätzlich wird falls vorhanden die Kurzinfo und Beschreibung aus
 * Konfiguration angezeigt.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class SystemObjectToolTipFigure extends Panel {

	public enum TooltipPlaceholders {
		/** Name des Systemobjekts. */
		NAME("{name}"),

		/** PID des Systemobjekts. */
		PID("{pid}"),

		/** Kurzinfo des Systemobjekts. */
		INFO("{info}"),

		/** Kurzinfo des Systemobjekts mit vorgestelltem Newline. */
		NL_INFO("{nlinfo}"),

		/** Beschreibung des Objekts. */
		DESC("{desc}"),

		/** Beschreibung des Objekts mit vorgestelltem Newline. */
		NL_DESC("{nldesc}"),

		/** Beschreibung des Objekts. */
		NEWLINE("{nl}");

		private String id;

		private TooltipPlaceholders(final String id) {
			this.id = id;
		}

	}

	private Label customNameLabel;
	private final Label nameLabel;
	private Label pidLabel;
	private Label kurzinfoLabel;
	private Label beschreibungLabel;

	private boolean infoRead;
	private String info;
	private String desc;

	/**
	 * Initialisiert den Tooltip. Das Systemobjekt muss später mit
	 * {@link #setSystemObject(SystemObject)} gesetzt werden.
	 */
	public SystemObjectToolTipFigure() {
		this(null);
	}

	/**
	 * Initialisiert den Tooltip.
	 *
	 * @param so
	 *            das Systemobjekt für das der Tooltip bestimmt ist.
	 */
	public SystemObjectToolTipFigure(final SystemObject so) {
		setLayoutManager(new GridLayout());

		nameLabel = new Label();
		add(nameLabel);

		setSystemObject(so);
	}

	/**
	 * Übernimmt den Namen und die PID aus dem Systemobjekt und initialisiert
	 * damit den Tooltip.
	 *
	 * @param systemObject
	 *            das Systemobjekt für das der Tooltip bestimmt ist.
	 */
	public void setSystemObject(final SystemObject systemObject) {

		infoRead = false;
		if (systemObject == null) {
			nameLabel.setText("Kein Systemobjekt gesetzt.");

			if (pidLabel != null) {
				remove(pidLabel);
				pidLabel = null;
			}

			if (kurzinfoLabel != null) {
				remove(kurzinfoLabel);
				kurzinfoLabel = null;
			}

			if (beschreibungLabel != null) {
				remove(beschreibungLabel);
				beschreibungLabel = null;
			}
		} else {

			final String defaultFormat = DObjPlugin.getDefault().getPreferenceStore()
					.getString(DobjPreference.TOOLTIP_FORMAT.getId());
			if ((defaultFormat != null) && !defaultFormat.isEmpty()) {
				formatTooltip(systemObject, defaultFormat);
				return;
			}

			nameLabel.setText(systemObject.getName());

			if (pidLabel == null) {
				pidLabel = new Label();
				add(pidLabel);
			}
			pidLabel.setText("(PID: " + systemObject.getPid() + ")");

			final AttributeGroup atg = systemObject.getDataModel().getAttributeGroup("atg.info");
			final Data data = systemObject.getConfigurationData(atg);
			if (data != null) {
				final String kurzinfo = data.getTextValue("kurzinfo").getText();
				if (!kurzinfo.isEmpty()) {
					if (kurzinfoLabel == null) {
						kurzinfoLabel = new Label();
						add(kurzinfoLabel);
					}
					kurzinfoLabel.setText(kurzinfo);
				}

				final String beschreibung = data.getTextValue("beschreibung").getText();
				if (!beschreibung.isEmpty()) {
					if (beschreibungLabel == null) {
						beschreibungLabel = new Label();
						add(beschreibungLabel);
					}
					beschreibungLabel.setText(beschreibung);
				}
			}
		}
	}

	private void formatTooltip(final SystemObject systemObject, final String formatStr) {

		String tooltipStr = formatStr;

		for (final TooltipPlaceholders item : TooltipPlaceholders.values()) {
			if (tooltipStr.contains(item.id)) {
				String replacement;
				switch (item) {
				case DESC:
					replacement = getDesc(systemObject);
					if (replacement == null) {
						replacement = "";
					}
					break;
				case INFO:
					replacement = getInfo(systemObject);
					if (replacement == null) {
						replacement = "";
					}
					break;
				case NAME:
					replacement = systemObject.getNameOrPidOrId();
					if (replacement == null) {
						replacement = "";
					}
					break;
				case NEWLINE:
					replacement = "\n";
					break;
				case NL_DESC:
					replacement = getDesc(systemObject);
					if (replacement != null) {
						replacement = "\n" + replacement;
					} else {
						replacement = "";
					}
					break;
				case NL_INFO:
					replacement = getInfo(systemObject);
					if (replacement != null) {
						replacement = "\n" + replacement;
					} else {
						replacement = "";
					}
					break;
				case PID:
					replacement = systemObject.getPid();
					if (replacement == null) {
						replacement = "";
					}
					break;
				default:
					replacement = null;
					break;
				}

				if (replacement != null) {
					while (tooltipStr.contains(item.id)) {
						tooltipStr = tooltipStr.replace(item.id, replacement);
					}
				}
			}
		}

		nameLabel.setText(tooltipStr);
	}

	private String getInfo(final SystemObject systemObject) {
		readInfo(systemObject);
		return info;
	}

	private String getDesc(final SystemObject systemObject) {
		readInfo(systemObject);
		return desc;
	}

	private void readInfo(final SystemObject systemObject) {

		if (infoRead) {
			return;
		}

		final AttributeGroup atg = systemObject.getDataModel().getAttributeGroup("atg.info");
		final Data data = systemObject.getConfigurationData(atg);
		infoRead = true;
		info = null;
		desc = null;

		if (data != null) {
			final String kurzinfo = data.getTextValue("kurzinfo").getText();
			if (!kurzinfo.isEmpty()) {
				info = kurzinfo;
			}

			final String beschreibung = data.getTextValue("beschreibung").getText();
			if (!beschreibung.isEmpty()) {
				desc = beschreibung;
			}
		}
	}

	/**
	 * Legt einen benutzerdefinierten Namen für das Objekt fest. Dieser wird in
	 * der ersten Zeile des Tooltips angezeigt.
	 *
	 * @param name
	 *            der benutzerdefiniert Name oder <code>null</code>, wenn keiner
	 *            angezeigt werden soll.
	 */
	public void setName(final String name) {
		if (name != null && !name.isEmpty()) {
			if (customNameLabel == null) {
				customNameLabel = new Label();
				add(customNameLabel, 0);
			}

			customNameLabel.setText(name);
		} else if (customNameLabel != null) {
			remove(customNameLabel);
			customNameLabel = null;
		}
	}

}
