/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.kollision;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.draw2d.geometry.Point;

/**
 * Verortet Objekte vom Typ {@link IArrangierbar} in einer Karte.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
public class KollisionsManager {

	private final Set<IArrangierbar> kollisionsTabelle = new HashSet<>();

	/**
	 * Verortet ein Objekt <code>objekt</code> in der Karte ohne das das Objekt
	 * mit anderen kollidiert.
	 *
	 * @param objekt
	 *            ein Objekt.
	 * @return der beste Punkt an dem das Objekt verortet werden kann.
	 */
	public final Point verorte(final IArrangierbar objekt) {
		synchronized (kollisionsTabelle) {
			if (!kollisionsTabelle.contains(objekt)) {
				kollisionsTabelle.add(objekt);
			}
		}
		double minKoeff = Double.MAX_VALUE;
		Point zielPunkt = objekt.getAlternativeOrte().iterator().next();

		if (objekt.isSichtbar()) {
			for (final Point moeglicherPunkt : objekt.getAlternativeOrte()) {
				synchronized (kollisionsTabelle) {
					double maxKoeffAufMoeglichemPunkt = IUeberdeckungsFunktion.KEINE_UEBERDECKUNG;
					for (final IArrangierbar arrangierbaresObjekt : kollisionsTabelle) {
						if (arrangierbaresObjekt.isSichtbar()) {
							if (objekt != arrangierbaresObjekt) {
								if (KollisionsUtil.haveSameScale(objekt, arrangierbaresObjekt)) {
									final IUeberdeckungsFunktion f = UeberdeckungsFunktionsProvider
											.getUeberdeckungsFunktionFuer(objekt, arrangierbaresObjekt);
									if (f != null) {
										final double koeff = f.getUeberdeckungsKoeffizient(moeglicherPunkt, objekt,
												arrangierbaresObjekt);
										if (koeff > maxKoeffAufMoeglichemPunkt) {
											maxKoeffAufMoeglichemPunkt = koeff;
										}
										if (maxKoeffAufMoeglichemPunkt >= IUeberdeckungsFunktion.VOLLSTAENDIGE_UEBERDECKUNG) {
											break;
										}
									}
								}
							}
						}
					}
					if (maxKoeffAufMoeglichemPunkt < minKoeff) {
						minKoeff = maxKoeffAufMoeglichemPunkt;
						zielPunkt = moeglicherPunkt;
					}
					if (minKoeff == IUeberdeckungsFunktion.KEINE_UEBERDECKUNG) {
						return zielPunkt;
					}
				}
			}
		}
		return zielPunkt;
	}

	/**
	 * Entfernt ein Objekt aus der Menge der Objekte, die nicht miteinander
	 * kollidieren sollen.
	 *
	 * @param objekt
	 *            ein Objekt.
	 */
	public final void remove(final IArrangierbar objekt) {
		synchronized (kollisionsTabelle) {
			kollisionsTabelle.remove(objekt);
		}
	}

}
