/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.kollision;

import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;

/**
 * Standardueberdeckungsfunktion. Errechnet den Ueberdeckungsvergleicht anhand
 * der Bounds der Figures.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 */
final class StandardUeberdeckungsFunktion implements IUeberdeckungsFunktion {

	static final StandardUeberdeckungsFunktion INSTANZ = new StandardUeberdeckungsFunktion();

	@Override
	public double getUeberdeckungsKoeffizient(final Point o1Ort, final Object o1, final Object o2) {
		final IFigure f1 = Platform.getAdapterManager().getAdapter(o1, IFigure.class);
		final IFigure f2 = Platform.getAdapterManager().getAdapter(o2, IFigure.class);
		if (f1 != null && f2 != null) {
			final Rectangle b1 = f1.getBounds();
			final Rectangle b2 = f2.getBounds();
			return getRectangleUeberdeckungsKoeffizient(o1Ort, b1, b2);
		}
		return KEINE_UEBERDECKUNG;
	}

	/**
	 * Berechnet den Ueberdeckungskoeffizienten von zwei Rechtecken.
	 *
	 * @param o1Ort
	 *            der Ort, an dem sich das Rechteck <code>b1</code> eigentlich
	 *            befinden soll.
	 * @param b1Vorlaeufig
	 *            ein Rechteck (nur Groesse von Bedeutung).
	 * @param b2
	 *            ein Rechteck.
	 * @return der Ueberdeckungskoeffizient.
	 */
	public static double getRectangleUeberdeckungsKoeffizient(final Point o1Ort, final Rectangle b1Vorlaeufig,
			final Rectangle b2) {
		Rectangle b1 = b1Vorlaeufig;
		if (o1Ort != null) {
			final Double newWidth = b1Vorlaeufig.width * 1.0;
			final Double newHeight = b1Vorlaeufig.height * 1.0;
			b1 = new Rectangle(o1Ort.x - new Double(newWidth / 2.0).intValue(),
					o1Ort.y - new Double(newHeight / 2.0).intValue(), newWidth.intValue(), newHeight.intValue());
		}
		if (b1 != null && b2 != null) {
			final Rectangle ueberdeckung = b1.getIntersection(b2);
			if (!ueberdeckung.isEmpty()) {
				final double b1A = b1.preciseWidth() * b1.preciseHeight();
				final double b2A = b2.preciseWidth() * b2.preciseHeight();
				final double ueberdeckungA = ueberdeckung.preciseWidth() * ueberdeckung.preciseHeight();
				return Math.max(ueberdeckungA / b1A, ueberdeckungA / b2A);
			}
		}
		return KEINE_UEBERDECKUNG;
	}

}
