/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.properties;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.emf.gef.model.GefPackage;
import com.bitctrl.lib.eclipse.emf.gef.model.Located;

import de.bsvrz.buv.plugin.dobj.util.DobjIcons;

/**
 * Tabbed Property Section für {@link Located}-Objekte.
 *
 * <p>
 * Bearbeitbar sind die folgenden Eingenschaften:
 * <ul>
 * <li>X-Position und</li>
 * <li>Y-Position</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class LocatedSection extends AbstractSection<Located> {

	private CLabel xLabel;
	private Spinner xSpinner;
	private CLabel yLabel;
	private Spinner ySpinner;

	@Override
	public void createControls(final Composite parent, final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createLocation(container);
	}

	private void createLocation(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Button resetToDefaultButton = new Button(parent, SWT.TOGGLE);
		resetToDefaultButton.setToolTipText("Defaultwert für die Position verwenden.");
		final Image image = DobjIcons.ActionRestoreDefault.getImageDescriptor().createImage();
		resetToDefaultButton.setImage(image);
		resetToDefaultButton.addDisposeListener(new DisposeListener() {

			@Override
			public void widgetDisposed(final DisposeEvent e) {
				image.dispose();
			}

		});
		data = new FormData();
		data.width = resetToDefaultButton.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		resetToDefaultButton.setLayoutData(data);

		xSpinner = new Spinner(parent, SWT.BORDER);
		xSpinner.setMinimum(Integer.MIN_VALUE);
		xSpinner.setMaximum(Integer.MAX_VALUE);
		factory.adapt(xSpinner);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(50, 0);
		data.top = new FormAttachment(resetToDefaultButton, 0, SWT.CENTER);
		xSpinner.setLayoutData(data);

		xLabel = factory.createCLabel(parent, "X-Position:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(xSpinner, -ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(xSpinner, 0, SWT.CENTER);
		xLabel.setLayoutData(data);

		ySpinner = new Spinner(parent, SWT.BORDER);
		ySpinner.setMinimum(Integer.MIN_VALUE);
		ySpinner.setMaximum(Integer.MAX_VALUE);
		factory.adapt(ySpinner);
		data = new FormData();
		data.left = new FormAttachment(xSpinner, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(resetToDefaultButton);
		data.top = new FormAttachment(xSpinner, 0, SWT.CENTER);
		ySpinner.setLayoutData(data);

		yLabel = factory.createCLabel(parent, "Y-Position:");
		data = new FormData();
		data.left = new FormAttachment(50, ITabbedPropertyConstants.HSPACE);
		data.right = new FormAttachment(ySpinner, -ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(ySpinner, 0, SWT.CENTER);
		yLabel.setLayoutData(data);

		// Listener aktivieren
		resetToDefaultButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean useDefault = resetToDefaultButton.getSelection();

				final Object value;
				if (useDefault) {
					value = null;
				} else {
					value = new Point();
				}

				final SetCommand cmd = new SetCommand(getElement(), GefPackage.Literals.LOCATED__LOCATION, value);
				getCommandStack().execute(cmd);

				setUseDefault(useDefault);
			}

		});
		final SelectionListener selectionListener = new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				EStructuralFeature feature;
				Object value;

				feature = GefPackage.Literals.LOCATED__LOCATION;
				if (e.widget == xSpinner) {
					value = new Point(xSpinner.getSelection(), getElement().getLocation().y);
				} else if (e.widget == ySpinner) {
					value = new Point(getElement().getLocation().x, ySpinner.getSelection());
				} else {
					return;
				}

				final SetCommand cmd = new SetCommand(getElement(), feature, value);
				getCommandStack().execute(cmd);
			}

		};
		xSpinner.addSelectionListener(selectionListener);
		ySpinner.addSelectionListener(selectionListener);
	}

	@Override
	public void refresh() {

		final Point location = getElement().getLocation();
		final int x;
		final int y;

		setUseDefault(location == null);
		if (location != null) {
			x = location.x;
			y = location.y;
		} else {
			x = 0;
			y = 0;
		}

		if (x != xSpinner.getSelection()) {
			xSpinner.setSelection(x);
		}
		if (y != ySpinner.getSelection()) {
			ySpinner.setSelection(y);
		}
	}

	private void setUseDefault(final boolean useDefault) {
		xLabel.setEnabled(!useDefault);
		xSpinner.setEnabled(!useDefault);
		yLabel.setEnabled(!useDefault);
		ySpinner.setEnabled(!useDefault);
	}

}
