/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.properties;

import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.bitctrl.lib.eclipse.emf.gef.commands.SetCommand;
import com.bitctrl.lib.eclipse.emf.gef.model.GefPackage;
import com.bitctrl.lib.eclipse.emf.gef.model.Sized;

import de.bsvrz.buv.plugin.dobj.util.DobjIcons;

/**
 * Tabbed Property Section für {@link Sized}-Objekte.
 *
 * <p>
 * Bearbeitbar sind die folgenden Eingenschaften:
 * <ul>
 * <li>Höhe und</li>
 * <li>Breite</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class SizedSection extends AbstractSection<Sized> {

	private CLabel widthLabel;
	private Spinner widthSpinner;
	private CLabel heightLabel;
	private Spinner heightSpinner;

	@Override
	public void createControls(final Composite parent, final TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		final Composite container = factory.createFlatFormComposite(parent);

		createLocation(container);
	}

	private void createLocation(final Composite parent) {
		final TabbedPropertySheetWidgetFactory factory = getWidgetFactory();
		FormData data;

		final Button resetToDefaultButton = new Button(parent, SWT.TOGGLE);
		resetToDefaultButton.setToolTipText("Defaultwert für die Größe verwenden.");
		final Image image = DobjIcons.ActionRestoreDefault.getImageDescriptor().createImage();
		resetToDefaultButton.setImage(image);
		resetToDefaultButton.addDisposeListener(new DisposeListener() {

			@Override
			public void widgetDisposed(final DisposeEvent e) {
				image.dispose();
			}

		});
		data = new FormData();
		data.width = resetToDefaultButton.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
		resetToDefaultButton.setLayoutData(data);

		widthSpinner = new Spinner(parent, SWT.BORDER);
		widthSpinner.setMinimum(1);
		widthSpinner.setMaximum(Integer.MAX_VALUE);
		widthSpinner.setIncrement(100);
		widthSpinner.setPageIncrement(1000);
		factory.adapt(parent);
		data = new FormData();
		data.left = new FormAttachment(0, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(50, 0);
		data.top = new FormAttachment(resetToDefaultButton, 0, SWT.CENTER);
		widthSpinner.setLayoutData(data);

		widthLabel = factory.createCLabel(parent, "Breite:");
		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(widthSpinner, -ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(widthSpinner, 0, SWT.CENTER);
		widthLabel.setLayoutData(data);

		heightSpinner = new Spinner(parent, SWT.BORDER);
		heightSpinner.setMinimum(1);
		heightSpinner.setMaximum(Integer.MAX_VALUE);
		heightSpinner.setIncrement(100);
		heightSpinner.setPageIncrement(1000);
		factory.adapt(parent);
		data = new FormData();
		data.left = new FormAttachment(widthSpinner, STANDARD_LABEL_WIDTH);
		data.right = new FormAttachment(resetToDefaultButton);
		data.top = new FormAttachment(widthSpinner, 0, SWT.CENTER);
		heightSpinner.setLayoutData(data);

		heightLabel = factory.createCLabel(parent, "Höhe:");
		data = new FormData();
		data.left = new FormAttachment(50, ITabbedPropertyConstants.HSPACE);
		data.right = new FormAttachment(heightSpinner, -ITabbedPropertyConstants.HSPACE);
		data.top = new FormAttachment(heightSpinner, 0, SWT.CENTER);
		heightLabel.setLayoutData(data);

		// Listener aktivieren
		resetToDefaultButton.addSelectionListener(new SelectionAdapter() {

			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean useDefault = resetToDefaultButton.getSelection();

				final Object value;
				if (useDefault) {
					value = null;
				} else {
					value = new Dimension();
				}

				final SetCommand cmd = new SetCommand(getElement(), GefPackage.Literals.SIZED__SIZE, value);
				getCommandStack().execute(cmd);

				setUseDefault(useDefault);
			}

		});
		final SelectionListener sizeListener = new SelectionAdapter() {

			@Override
			public void widgetSelected(final org.eclipse.swt.events.SelectionEvent e) {
				final EStructuralFeature feature;
				final Object value;

				if (e.widget == widthSpinner || e.widget == heightSpinner) {
					Dimension size = getElement().getSize();
					if (size == null) {
						size = new Dimension();
					}

					feature = GefPackage.Literals.SIZED__SIZE;
					if (e.widget == widthSpinner) {
						size.width = widthSpinner.getSelection();
					} else {
						size.height = heightSpinner.getSelection();
					}

					value = size;
				} else {
					return;
				}

				final SetCommand cmd = new SetCommand(getElement(), feature, value);
				getCommandStack().execute(cmd);
			}

		};
		widthSpinner.addSelectionListener(sizeListener);
		heightSpinner.addSelectionListener(sizeListener);
	}

	@Override
	public void refresh() {
		final Dimension size = getElement().getSize();
		final int width;
		final int height;

		setUseDefault(size == null);
		if (size != null) {
			width = size.width;
			height = size.height;
		} else {
			width = 0;
			height = 0;
		}

		if (width != widthSpinner.getSelection()) {
			widthSpinner.setSelection(width);
		}
		if (height != heightSpinner.getSelection()) {
			heightSpinner.setSelection(height);
		}
	}

	private void setUseDefault(final boolean useDefault) {
		widthLabel.setEnabled(!useDefault);
		widthSpinner.setEnabled(!useDefault);
		heightLabel.setEnabled(!useDefault);
		heightSpinner.setEnabled(!useDefault);
	}

}
