/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.tools;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import org.eclipse.draw2d.Cursors;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.KeyHandler;
import org.eclipse.gef.Request;
import org.eclipse.gef.RequestConstants;
import org.eclipse.gef.tools.AbstractTool;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.events.KeyEvent;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.plugin.dobj.util.SelectionProperties;

public abstract class AbstractMarqueeSelectionTool extends BaseTool {

	protected static final Request MARQUEE_REQUEST = new Request(RequestConstants.REQ_SELECTION);

	protected static final int DEFAULT_MODE = 0;
	protected static final int TOGGLE_MODE = 1;
	protected static final int APPEND_MODE = 2;

	private int mode;
	private IFigure marqueeFeedbackFigure;
	private Set<EditPart> allChildren;
	private Collection<EditPart> selectedEditParts;
	private Request targetRequest;

	public AbstractMarqueeSelectionTool() {
		setUnloadWhenFinished(false);
		setDefaultCursor(Cursors.CROSS);
	}

	protected int getSelectionMode() {
		return mode;
	}

	protected void setSelectionMode(final int mode) {
		this.mode = mode;
	}

	protected boolean isTouchedSelection() {
		final SelectionProperties property = (SelectionProperties) getCurrentViewer()
				.getProperty(SelectionProperties.class.toString());
		if (property != null) {
			return property.isSelectTouched();
		}

		return false;
	}

	protected boolean isShowFeedback() {
		final SelectionProperties property = (SelectionProperties) getCurrentViewer()
				.getProperty(SelectionProperties.class.toString());
		if (property != null) {
			return property.isShowFeedback();
		}

		return false;
	}

	/**
	 * This method is called when mouse or keyboard input is invalid and erases
	 * the feedback.
	 *
	 * @return <code>true</code>
	 */
	@Override
	protected boolean handleInvalidInput() {
		eraseTargetFeedback();
		eraseMarqueeFeedback();
		return true;
	}

	private Request getTargetRequest() {
		if (targetRequest == null) {
			targetRequest = createTargetRequest();
		}
		return targetRequest;
	}

	private Request createTargetRequest() {
		return MARQUEE_REQUEST;
	}

	protected void eraseTargetFeedback() {
		if (selectedEditParts == null) {
			return;
		}
		final Iterator<EditPart> oldEditParts = selectedEditParts.iterator();
		while (oldEditParts.hasNext()) {
			final EditPart editPart = oldEditParts.next();
			editPart.eraseTargetFeedback(getTargetRequest());
		}
	}

	protected void eraseMarqueeFeedback() {
		if (marqueeFeedbackFigure != null) {
			removeFeedback(marqueeFeedbackFigure);
			marqueeFeedbackFigure = null;
		}
	}

	protected void performMarqueeSelect() {
		final EditPartViewer viewer = getCurrentViewer();
		final Collection<EditPart> newSelections = new LinkedHashSet<>();
		final Collection<EditPart> deselections = new HashSet<>();
		calculateNewSelection(newSelections, deselections);
		if (getSelectionMode() != DEFAULT_MODE) {
			newSelections.addAll(viewer.getSelectedEditParts());
			newSelections.removeAll(deselections);
		}
		viewer.setSelection(new StructuredSelection(newSelections.toArray()));
	}

	protected abstract void calculateNewSelection(final Collection<EditPart> newSelections,
			final Collection<EditPart> deselections);

	protected Set<EditPart> getAllChildren() {
		if (allChildren == null) {
			allChildren = DobjUtil.getAllChildren(getCurrentViewer().getRootEditPart());
		}
		return allChildren;
	}

	/**
	 * Prüft das Modellobjekt eines Edit Parts. Es ist valid, wenn es ein
	 * {@link DoModel} ist und wenn {@link SelectionProperties} vorhanden sind,
	 * muss es in der Matchingmenge enthalten sein.
	 */
	protected boolean isValidModel(final Object object) {
		final SelectionProperties selectionProperties = (SelectionProperties) getCurrentViewer()
				.getProperty(SelectionProperties.class.toString());

		if (object instanceof DoModel) {
			if (selectionProperties == null) {
				return true;
			}

			final DoModel model = (DoModel) object;
			return selectionProperties.getMatchDoTypen().contains(model.getDoTyp());
		}

		return false;
	}

	protected boolean isFigureVisible(final IFigure fig) {
		final Rectangle figBounds = fig.getBounds().getCopy();
		IFigure walker = fig.getParent();
		while (!figBounds.isEmpty() && walker != null) {
			walker.translateToParent(figBounds);
			figBounds.intersect(walker.getBounds());
			walker = walker.getParent();
		}
		return !figBounds.isEmpty();
	}

	protected IFigure getMarqueeFeedbackFigure() {
		if (marqueeFeedbackFigure == null) {
			marqueeFeedbackFigure = createMarqueeFeedbackFigure();
			addFeedback(marqueeFeedbackFigure);
		}
		return marqueeFeedbackFigure;
	}

	protected abstract IFigure createMarqueeFeedbackFigure();

	protected abstract void showMarqueeFeedback();

	protected void showTargetFeedback() {
		for (final Iterator<EditPart> itr = selectedEditParts.iterator(); itr.hasNext();) {
			final EditPart editPart = itr.next();
			editPart.showTargetFeedback(getTargetRequest());
		}
	}

	/**
	 * Handles high-level processing of a key down event. KeyEvents are
	 * forwarded to the current viewer's {@link KeyHandler}, via
	 * {@link KeyHandler#keyPressed(KeyEvent)}.
	 *
	 * @see AbstractTool#handleKeyDown(KeyEvent)
	 */
	@Override
	protected boolean handleKeyDown(final KeyEvent e) {
		if (super.handleKeyDown(e)) {
			return true;
		}
		if (getCurrentViewer().getKeyHandler() != null) {
			return getCurrentViewer().getKeyHandler().keyPressed(e);
		}
		return false;
	}

	protected Collection<EditPart> getSelectedEditParts() {
		return selectedEditParts;
	}

	protected void clearSelectedEditParts() {
		selectedEditParts = new ArrayList<>();
	}

	protected void clearAllChildren() {
		allChildren = null;
	}

}
