/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.tools;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.draw2d.FigureCanvas;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.gef.DragTracker;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.RootEditPart;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.tools.PanningSelectionTool;
import org.eclipse.ui.part.EditorPart;

import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.dobj.util.DobjUtil;
import de.bsvrz.buv.plugin.dobj.util.SelectionProperties;

/**
 * Standardselection-Tool mit der Eigenschaft, dass uebereinanderliegende
 * Objekte einzeln zur Auswahl angeboten werden.
 *
 * @author BitCtrl Systems GmbH, Thierfelder
 *
 */
public class DObjSelectionTool extends PanningSelectionTool {

	private Set<EditPart> allChildren;

	@Override
	protected boolean handleButtonDown(final int button) {
		final EditPart editpart = getTargetEditPart();
		if (button == 1 && editpart != null) {
			if (isValidModel(editpart.getModel())) {
				final Set<EditPart> moeglicheObjekteInAusschnitt = getCrowdedEditPartsNear(editpart);
				if (moeglicheObjekteInAusschnitt.size() > 1) {
					final DragTracker tracker = new CrowdedEditPartsTracker(editpart, moeglicheObjekteInAusschnitt);
					tracker.setViewer(getCurrentViewer());
					setDragTracker(tracker);
					lockTargetEditPart(editpart);
					return true;
				}
			} else {
				if (editpart.getModel() == null || !(editpart.getModel() instanceof DoModel)) {
					return super.handleButtonDown(button);
				}
				if (button == 1 && getCurrentViewer().getControl() instanceof FigureCanvas
						&& stateTransition(PAN, PAN_IN_PROGRESS)) {
					return super.handleButtonDown(button);
				}
				return true;
			}
		}

		return super.handleButtonDown(button);
	}

	/**
	 * Erfragt alle {@link GraphicalEditPart}s, die in der Naehe vom
	 * uebergebenen {@link GraphicalEditPart} liegen, bzw. von diesem ueberdeckt
	 * werden.
	 *
	 * @param sourceEditPart
	 *            ein selektierter {@link GraphicalEditPart}.
	 * @return die Menge aller {@link EditorPart}s, die in der Naehe vom
	 *         uebergebenen {@link GraphicalEditPart} liegen, bzw. von diesem
	 *         ueberdeckt werden.
	 */
	private Set<EditPart> getCrowdedEditPartsNear(final EditPart sourceEditPart) {
		final Set<EditPart> crowdedEditParts = new HashSet<>();

		if (getCurrentViewer() != null && getCurrentViewer() instanceof GraphicalViewer
				&& getCurrentViewer().getRootEditPart() != null) {
			final ZoomManager zm = (ZoomManager) getCurrentViewer().getProperty(ZoomManager.class.toString());
			if (zm != null) {
				Point drawPoint = getLocation();
				final RootEditPart root = getCurrentViewer().getRootEditPart();
				if (root != null && root instanceof ScalableRootEditPart) {
					final ScalableRootEditPart scalableRoot = (ScalableRootEditPart) root;
					drawPoint = new Point(drawPoint.x + scalableRoot.getFigure().getClientArea().x,
							drawPoint.y + scalableRoot.getFigure().getClientArea().y);
				}
				drawPoint.performScale(1.0 / zm.getZoom());

				final Set<EditPart> moeglicheObjekteInAusschnitt = getAllChildren();
				for (final EditPart e : moeglicheObjekteInAusschnitt) {
					final GraphicalEditPart target = (GraphicalEditPart) e;
					if (!(target instanceof RootEditPart) && target.getFigure().isShowing()) {
						if (!isValidModel(target.getModel())) {
							continue;
						}

						if (target.getFigure().containsPoint(drawPoint) && target.isSelectable()) {
							crowdedEditParts.add(target);
						}
					}
				}
			}
		}

		return crowdedEditParts;
	}

	/**
	 * Prüft das Modellobjekt eines Edit Parts. Es ist valid, wenn es ein
	 * {@link DoModel} ist und wenn {@link SelectionProperties} vorhanden sind,
	 * muss es in der Matchingmenge enthalten sein.
	 */
	protected boolean isValidModel(final Object object) {
		final SelectionProperties selectionProperties = (SelectionProperties) getCurrentViewer()
				.getProperty(SelectionProperties.class.toString());

		if (object instanceof DoModel) {
			if (selectionProperties == null) {
				return true;
			}

			final DoModel model = (DoModel) object;
			return selectionProperties.getMatchDoTypen().contains(model.getDoTyp());
		}

		return false;
	}

	/**
	 * Gibt alle Edit Parts unterhalb des Root Edit Parts zurück.
	 */
	protected Set<EditPart> getAllChildren() {
		if (allChildren == null) {
			allChildren = DobjUtil.getAllChildren(getCurrentViewer().getRootEditPart());
		}
		return allChildren;
	}

	/** setzt die Liste der gesicherten Kind-EditParts zurück. */
	public void resetChildren() {
		allChildren = null;
	}
}
