/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.tools;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.swt.SWT;

import de.bsvrz.buv.plugin.dobj.figures.RectangleFeedbackFigure;

public class RectangleSelectionTool extends AbstractMarqueeSelectionTool {

	@Override
	protected String getCommandName() {
		return "Rechteckselektion";
	}

	/**
	 * Erases feedback if necessary and puts the tool into the terminal state.
	 */
	@Override
	public void deactivate() {
		if (isInState(STATE_DRAG_IN_PROGRESS)) {
			eraseMarqueeFeedback();
			eraseTargetFeedback();
		}
		super.deactivate();
		clearAllChildren();
		setState(STATE_TERMINAL);
	}

	@Override
	protected boolean handleButtonDown(final int button) {
		if (!isGraphicalViewer()) {
			return true;
		}
		if (button != 1) {
			setState(STATE_INVALID);
			handleInvalidInput();
		}
		if (stateTransition(STATE_INITIAL, STATE_DRAG_IN_PROGRESS)) {
			if (getCurrentInput().isModKeyDown(SWT.MOD1)) {
				setSelectionMode(TOGGLE_MODE);
			} else if (getCurrentInput().isShiftKeyDown()) {
				setSelectionMode(APPEND_MODE);
			} else {
				setSelectionMode(DEFAULT_MODE);
			}
		}
		return true;
	}

	/**
	 * @see org.eclipse.gef.tools.AbstractTool#handleButtonUp(int)
	 */
	@Override
	protected boolean handleButtonUp(final int button) {
		if (stateTransition(STATE_DRAG_IN_PROGRESS, STATE_TERMINAL)) {
			eraseTargetFeedback();
			eraseMarqueeFeedback();
			performMarqueeSelect();
		}
		handleFinished();
		return true;
	}

	/**
	 * @see org.eclipse.gef.tools.AbstractTool#handleDragInProgress()
	 */
	@Override
	protected boolean handleDragInProgress() {
		if (isInState(STATE_DRAG | STATE_DRAG_IN_PROGRESS)) {
			showMarqueeFeedback();
			eraseTargetFeedback();
			clearSelectedEditParts();
			calculateNewSelection(getSelectedEditParts(), new ArrayList<EditPart>());
			showTargetFeedback();

			if (isShowFeedback()) {
				performMarqueeSelect();
			}
		}
		return true;
	}

	/**
	 * @see org.eclipse.gef.tools.AbstractTool#handleFocusLost()
	 */
	@Override
	protected boolean handleFocusLost() {
		if (isInState(STATE_DRAG | STATE_DRAG_IN_PROGRESS)) {
			handleFinished();
			return true;
		}
		return false;
	}

	@Override
	protected void calculateNewSelection(final Collection<EditPart> newSelections,
			final Collection<EditPart> deselections) {
		final Rectangle marqueeRect = getMarqueeSelectionRectangle();

		for (final Iterator<EditPart> itr = getAllChildren().iterator(); itr.hasNext();) {
			final GraphicalEditPart child = (GraphicalEditPart) itr.next();
			if (!isValidModel(child.getModel())) {
				continue;
			}

			final IFigure figure = child.getFigure();
			if (!child.isSelectable() || child.getTargetEditPart(MARQUEE_REQUEST) != child || !isFigureVisible(figure)
					|| !figure.isShowing()) {
				continue;
			}

			final Rectangle r = figure.getBounds().getCopy();
			figure.translateToAbsolute(r);
			if (isTouchedSelection()) {
				if (marqueeRect.contains(r) || marqueeRect.intersects(r)) {
					if (child.getSelected() == EditPart.SELECTED_NONE || getSelectionMode() != TOGGLE_MODE) {
						newSelections.add(child);
					} else {
						deselections.add(child);
					}
				}
			} else {
				if (marqueeRect.contains(r)) {
					if (child.getSelected() == EditPart.SELECTED_NONE || getSelectionMode() != TOGGLE_MODE) {
						newSelections.add(child);
					} else {
						deselections.add(child);
					}
				}
			}
		}
	}

	protected Rectangle getMarqueeSelectionRectangle() {
		return new Rectangle(getStartLocation(), getLocation());
	}

	@Override
	protected void showMarqueeFeedback() {
		final Rectangle rect = getMarqueeSelectionRectangle().getCopy();
		getMarqueeFeedbackFigure().translateToRelative(rect);
		getMarqueeFeedbackFigure().setBounds(rect);
	}

	@Override
	protected IFigure createMarqueeFeedbackFigure() {
		return new RectangleFeedbackFigure();
	}

}
