/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.tools;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.eclipse.core.runtime.Platform;
import org.eclipse.draw2d.Cursors;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.EditPartViewer.Conditional;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.SharedCursors;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;

import com.bitctrl.graph.Bogen;
import com.bitctrl.graph.GraphAlgorithm;
import com.bitctrl.util.CollectionUtilities;

import de.bsvrz.buv.plugin.dobj.figures.LineFeedbackFigure;
import de.bsvrz.buv.plugin.dobj.graph.NetzBogen;
import de.bsvrz.buv.plugin.dobj.graph.NetzGraph;
import de.bsvrz.buv.plugin.dobj.graph.NetzKnoten;
import de.bsvrz.buv.plugin.dobj.internal.DObjPlugin;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.model.BitCtrlDoModel;
import de.bsvrz.buv.plugin.dobj.model.DoModel;
import de.bsvrz.buv.plugin.selektion.SelektionsTransfer;
import de.bsvrz.buv.rw.bitctrl.BitCtrlRahmenwerkConstants;
import de.bsvrz.buv.rw.bitctrl.CacheService;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.NetzCacheExtended;

/**
 * Tool zum Selektieren einer Strecke. Eine Strecke besteht aus einer Menge von
 * aufeinanderfolgenden Straßensegmenten zwischen zwei Straßenknoten.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public class StreckeSelectionTool extends AbstractDragTool {

	private static final Conditional CONDITIONAL_STRASSENKNOTEN = new EditPartViewer.Conditional() {

		@Override
		public boolean evaluate(final EditPart editpart) {
			final SystemObject obj = Platform.getAdapterManager().getAdapter(editpart, SystemObject.class);
			if (obj != null) {
				return obj.isOfType(StrassenKnoten.PID);
			}
			return false;
		}

	};

	private LineFeedbackFigure feedbackFigure;
	private NetzGraph graph;
	private NetzKnoten startKnoten;
	private NetzKnoten zielKnoten;
	private boolean initialized;
	private final CacheService cacheService = RahmenwerkService.getService().getCacheService();

	private final PropertyChangeListener cacheListener = new PropertyChangeListener() {

		@Override
		public void propertyChange(final PropertyChangeEvent evt) {
			initialized = cacheService.getNetzCacheExtended(cacheService.getDefaultNetzPid()).isInitialisiert();
		}

	};

	/**
	 * Initialisiert das Tool.
	 */
	public StreckeSelectionTool() {
		setDefaultCursor(Cursors.CROSS);
		setUnloadWhenFinished(false);
	}

	@Override
	protected String getCommandName() {
		return "Streckenselektion";
	}

	@Override
	public void activate() {
		final NetzCacheExtended nce = cacheService.getNetzCacheExtended(cacheService.getDefaultNetzPid());
		initialized = nce.isInitialisiert();

		nce.addPropertyChangeListener(cacheListener);

		super.activate();

		graph = new NetzGraph(nce.getNetz());
	}

	@Override
	public void deactivate() {
		super.deactivate();
		final NetzCacheExtended nce = cacheService.getNetzCacheExtended(cacheService.getDefaultNetzPid());
		nce.removePropertyChangeListener(cacheListener);
	}

	@Override
	protected boolean handleMove() {
		if (getStrassenknoten(getLocation()) != null) {
			setCursor(Cursors.CROSS);
		} else {
			setCursor(SharedCursors.CURSOR_PLUG_NOT);
		}
		return true;
	}

	private EditPart getStrassenknoten(final Point location) {
		final GraphicalViewer viewer = (GraphicalViewer) getCurrentViewer();
		final EditPart editPart = viewer.findObjectAtExcluding(location, Collections.emptyList(),
				CONDITIONAL_STRASSENKNOTEN);
		final SystemObject so = Platform.getAdapterManager().getAdapter(editPart, SystemObject.class);

		if (so != null && so.isOfType(StrassenKnoten.PID)) {
			return editPart;
		}

		return null;
	}

	@Override
	protected boolean handleButtonDown(final int button) {
		if (!initialized) {
			MessageDialog.openInformation(DObjPlugin.getDefault().getWorkbench().getDisplay().getActiveShell(),
					"Streckenselektion ist noch nicht initialisiert.",
					"Bitte Warten. Das Streckenselektionswerkzeug kann erst verwendet werden, wenn der erweiterte Netz-Cache geladen ist.");
			return false;
		}
		if (!isGraphicalViewer()) {
			return true;
		}
		if (button != 1 || getStrassenknoten(getLocation()) == null) {
			setState(STATE_INVALID);
			handleInvalidInput();
		} else {
			initDrag();
		}
		stateTransition(STATE_INITIAL, STATE_DRAG_IN_PROGRESS);
		return true;
	}

	@Override
	protected boolean handleInvalidInput() {
		graph = null;
		startKnoten = null;
		zielKnoten = null;
		return super.handleInvalidInput();
	}

	private void initDrag() {
		final GraphicalViewer viewer = (GraphicalViewer) getCurrentViewer();
		final EditPart startEditPart = viewer.findObjectAtExcluding(getStartLocation(), Collections.emptySet(),
				CONDITIONAL_STRASSENKNOTEN);

		startKnoten = graph.getKnoten(((BitCtrlDoModel<StrassenKnoten>) startEditPart.getModel()).getSystemObjekt());

		// FIXME richtigen Algorithmus finden
		// breitensuche(graph, startKnoten, zielKnoten);
		GraphAlgorithm.breitensuche(graph, startKnoten);
		// GraphAlgorithm.dijkstra(graph, startKnoten);
	}

	@Override
	protected boolean handleDragInProgress() {
		if (startKnoten == null) {
			setCursor(SharedCursors.CURSOR_PLUG_NOT);
			return super.handleDragInProgress();
		}

		final EditPart zielEditPart = getStrassenknoten(getLocation());
		if (zielEditPart == null) {
			setCursor(SharedCursors.CURSOR_PLUG_NOT);
			return super.handleDragInProgress();
		}

		zielKnoten = graph.getKnoten(((BitCtrlDoModel<StrassenKnoten>) zielEditPart.getModel()).getSystemObjekt());

		final Set<String> intersection = getStrassenIdsVonStartNachZielKnoten();
		if (intersection.isEmpty()) {
			setCursor(SharedCursors.CURSOR_PLUG_NOT);
		} else {
			setCursor(Cursors.CROSS);
		}

		return super.handleDragInProgress();
	}

	@Override
	protected void handleFinished() {
		if (startKnoten == null || zielKnoten == null) {
			super.handleFinished();
			return;
		}

		final Set<String> strassen = getStrassenIdsVonStartNachZielKnoten();
		if (strassen.isEmpty()) {
			super.handleFinished();
			return;
		}

		final GraphicalViewer viewer = (GraphicalViewer) getCurrentViewer();
		final List<SystemObjekt> systemObjekte = new ArrayList<>();
		final List<Bogen> bogenListe = GraphAlgorithm.getPfadVonWurzel(zielKnoten);
		if (bogenListe == null) {
			// System.err.println("Knoten nicht erreichbar.");
		} else {
			final List<StrassenKnoten> knotenList = new ArrayList<>();
			knotenList.add(startKnoten.getStrassenKnoten());
			for (int i = 0; i < bogenListe.size(); ++i) {
				final NetzBogen bogen = (NetzBogen) bogenListe.get(i);
				final AeusseresStrassenSegment nachStrassenSegment = bogen.getAeusseresStrassenSegment();

				if (i > 0) {
					final NetzBogen lastBogen = (NetzBogen) bogenListe.get(i - 1);
					final AeusseresStrassenSegment vonStrassenSegment = lastBogen.getAeusseresStrassenSegment();
					final NetzKnoten knoten = (NetzKnoten) bogen.getAnfangsKnoten();
					knotenList.add(knoten.getStrassenKnoten());
					final InneresStrassenSegment iss = knoten.getInneresStrassenSegment(vonStrassenSegment,
							nachStrassenSegment);

					systemObjekte.add(iss);
				}
				systemObjekte.add(nachStrassenSegment);
			}
			knotenList.add(zielKnoten.getStrassenKnoten());

			String strassenIdTarget = null;
			if (!strassen.isEmpty()) {
				strassenIdTarget = strassen.iterator().next();
			}

			if (knotenList.size() > 1) {
				final NetzCacheExtended nce = cacheService.getNetzCacheExtended(cacheService.getDefaultNetzPid());
				final Map<StrassenKnoten, List<AeusseresStrassenSegment>> knotenToAssListMap = new HashMap<>();
				for (int k = 0; k < knotenList.size() - 1; k++) {
					final StrassenKnoten sk1 = knotenList.get(k);
					final StrassenKnoten sk2 = knotenList.get(k + 1);
					final List<AeusseresStrassenSegment> segmenteAusKnoten = nce.getAeussereStrassenSegmente(sk1);
					final List<AeusseresStrassenSegment> assList = new ArrayList<>();
					for (final AeusseresStrassenSegment ass : segmenteAusKnoten) {
						final StrassenKnoten nach = ass.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
						if (nach.equals(sk2)) {
							assList.add(ass);
						}
					}
					knotenToAssListMap.put(sk1, assList);
				}

				final List<SystemObjekt> objekteDummy = new ArrayList<>();
				if (strassenIdTarget != null) {
					for (int k = 0; k < knotenList.size() - 1; k++) {
						final StrassenKnoten sk1 = knotenList.get(k);
						final StrassenKnoten sk2 = knotenList.get(k + 1);
						final List<AeusseresStrassenSegment> assTargetList = new ArrayList<>();
						for (final AeusseresStrassenSegment ass : knotenToAssListMap.get(sk1)) {
							final String strassenId = getStrassenId(ass);
							if (strassenId != null && strassenId.equals(strassenIdTarget)) {
								assTargetList.add(ass);
							}
						}

						if (!assTargetList.isEmpty()) {
							AttTmcRichtung assRichtung = null;

							InneresStrassenSegment issTarget = null;
							boolean assTargetSet = false;
							for (final AeusseresStrassenSegment assTarget : assTargetList) {
								if (assTarget.getKdAeusseresStrassenSegment() != null
										&& assTarget.getKdAeusseresStrassenSegment().getDatum() != null && assTarget
												.getKdAeusseresStrassenSegment().getDatum().getTmcRichtung() != null) {
									assRichtung = assTarget.getKdAeusseresStrassenSegment().getDatum().getTmcRichtung();
								}

								if (!assTargetSet) {
									objekteDummy.add(assTarget);
									assTargetSet = true;
								}
								if (k < knotenList.size() - 2) {

									for (final InneresStrassenSegment iss : nce.getInneresStrassenSegmente(sk2,
											assTarget)) {

										final String issStrassenId = getStrassenId(iss);
										final AttTmcRichtung issRichtung = nce.getRichtung(iss);

										if (assRichtung != null && issRichtung != null) {
											if (issStrassenId != null && issStrassenId.equals(strassenIdTarget)
													&& assRichtung.equals(issRichtung)) {
												issTarget = iss;
												break;
											}
										} else {
											if (issStrassenId != null && issStrassenId.equals(strassenIdTarget)) {
												issTarget = iss;
												break;
											}
										}
									}
									if (issTarget != null) {
										break;
									}
								}
							}
							if (issTarget != null) {
								objekteDummy.add(issTarget);
							}
						}
					}
				}
				systemObjekte.clear();
				systemObjekte.addAll(objekteDummy);
			}
		}

		final Map<?, ?> editPartRegistry = viewer.getEditPartRegistry();
		final List<EditPart> editParts = new ArrayList<>();

		for (final SystemObjekt o : systemObjekte) {
			for (final Entry<?, ?> e : editPartRegistry.entrySet()) {
				if (e.getKey() instanceof DoModel) {
					final DoModel model = (DoModel) e.getKey();
					final EditPart editPart = (EditPart) e.getValue();
					if (editPart instanceof IStreckenSelektierbar) {
						if (model instanceof BitCtrlDoModel<?>) {
							if (o.equals(((BitCtrlDoModel<?>) model).getSystemObjekt())) {
								if (editPart.isSelectable()) {
									editParts.add(editPart);
								}
							}
						}
					}
				}
			}
		}

		viewer.setSelection(new StructuredSelection(editParts));
		if (editParts.size() > 0) {
			viewer.reveal(editParts.get(editParts.size() - 1));
		}

		// FIXME Selektion für Streckenprofil ohne festen Namen realisieren
		SelektionsTransfer.getInstanz().setDefinierteSelektion(BitCtrlRahmenwerkConstants.SELECTION_STRECKEN_ZUG, true,
				new StructuredSelection(systemObjekte));

		super.handleFinished();
	}

	private Set<String> getStrassenIdsVonStartNachZielKnoten() {
		Set<String> strassenIds = new HashSet<>();

		if (startKnoten != null && zielKnoten != null) {
			final Set<String> strassen1 = new HashSet<>();
			for (final Bogen b : startKnoten.ausgangsBogenIterator()) {
				final NetzBogen nb = (NetzBogen) b;
				final AeusseresStrassenSegment ass = nb.getAeusseresStrassenSegment();
				final String strassenId = getStrassenId(ass);
				if (strassenId != null) {
					strassen1.add(strassenId);
				}
			}

			final Set<String> strassen2 = new HashSet<>();
			for (final Bogen b : zielKnoten.ausgangsBogenIterator()) {
				final NetzBogen nb = (NetzBogen) b;
				final AeusseresStrassenSegment ass = nb.getAeusseresStrassenSegment();
				final String strassenId = getStrassenId(ass);
				if (strassenId != null) {
					strassen2.add(strassenId);
				}
			}

			strassenIds = CollectionUtilities.intersection(strassen1, strassen2);
		}

		return strassenIds;
	}

	private static String getStrassenId(
			final de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment ass) {
		String strassenId = null;

		if (ass.getKdStrassenSegment() != null) {
			final KdStrassenSegment.Daten datumStrassenSegment = ass.getKdStrassenSegment().getDatum();
			if (datumStrassenSegment != null && datumStrassenSegment.getGehoertZuStrasse() != null) {
				final Strasse strasse = datumStrassenSegment.getGehoertZuStrasse();
				if (strasse.getKdStrasse() != null && strasse.getKdStrasse().getDatum() != null) {
					strassenId = strasse.getKdStrasse().getDatum().getTyp().toString() + " "
							+ strasse.getKdStrasse().getDatum().getNummer() + " "
							+ strasse.getKdStrasse().getDatum().getZusatz();
					strassenId = strassenId.trim();
				}
			}
		}

		return strassenId;
	}

	private LineFeedbackFigure getFeedbackFigure() {
		if (feedbackFigure == null) {
			feedbackFigure = new LineFeedbackFigure();
			addFeedback(feedbackFigure);
		}
		return feedbackFigure;
	}

	@Override
	protected void showFeedback() {
		final Point start = getStartLocation();
		final Point end = getLocation();
		final Rectangle rect = new Rectangle(start, end);
		getFeedbackFigure().translateToRelative(rect);
		getFeedbackFigure().translateToRelative(start);
		getFeedbackFigure().translateToRelative(end);
		getFeedbackFigure().setStart(start);
		getFeedbackFigure().setEnd(end);
		getFeedbackFigure().setBounds(rect);
	}

	@Override
	protected void eraseFeedback() {
		if (feedbackFigure != null) {
			removeFeedback(feedbackFigure);
			feedbackFigure = null;
		}
	}

}
