/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.util;

import java.io.ByteArrayInputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.swing.event.EventListenerList;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.ImageData;

import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsEvent;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsListener;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.DynamicObjectType.DynamicObjectCreatedListener;
import de.bsvrz.dav.daf.main.config.DynamicObjectType.NameChangeListener;
import de.bsvrz.dav.daf.main.config.InvalidationListener;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.konfigurationsdaten.KdBild;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.objekte.Bild;
import de.bsvrz.sys.funclib.bitctrl.modell.metamodellglobal.attribute.AttByte_JavaKeyword;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;

/**
 * Verwaltet dynamische Objekte vom Typ <code>typ.bild</code>.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class BildManager implements DavVerbindungsListener {

	/** Die einzige Instanz des Bildmanagers. */
	public static final BildManager INSTANCE = new BildManager();

	private class InternalListener implements DynamicObjectCreatedListener, InvalidationListener, NameChangeListener {

		@Override
		public void invalidObject(final DynamicObject so) {
			fireBildEntfernt((Bild) RahmenwerkService.getService().getObjektFactory().getModellobjekt(so));
		}

		@Override
		public void objectCreated(final DynamicObject so) {
			fireBildAngelegt((Bild) RahmenwerkService.getService().getObjektFactory().getModellobjekt(so));
		}

		@Override
		public void nameChanged(final DynamicObject so) {
			fireBildUmbenannt((Bild) RahmenwerkService.getService().getObjektFactory().getModellobjekt(so));
		}
	}

	private final EventListenerList listener = new EventListenerList();
	private final InternalListener internalListener = new InternalListener();
	private boolean registered;

	/** speichert die Liste der unskalierten Imagedaten. */
	private final Map<Bild, ImageData> bilder = new HashMap<>();

	private BildManager() {
		RahmenwerkService.getService().getRahmenWerk().addDavVerbindungsListener(this);
	}

	/**
	 * Meldet einen Listener auf Änderungen an den vorhandenen Bildern ab.
	 */
	public synchronized void addBilderListener(final BildListener l) {
		listener.add(BildListener.class, l);
		registerDavListener();
	}

	private void registerDavListener() {
		if (!registered) {
			if (listener.getListenerCount(BildListener.class) == 1) {
				final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
				if (dav != null) {
					final DynamicObjectType type = (DynamicObjectType) dav.getDataModel().getType(Bild.PID);
					type.addObjectCreationListener(internalListener);
					type.addInvalidationListener(internalListener);
					type.addNameChangeListener(internalListener);
					registered = true;
				}
			}
		}
	}

	/**
	 * Meldet einen Listener auf Änderungen an den vorhandenen Bildern wieder
	 * ab.
	 */
	public synchronized void removeBilderListener(final BildListener l) {
		listener.remove(BildListener.class, l);
		removeDavListener(false);
	}

	private void removeDavListener(final boolean forced) {
		if (registered) {
			if (listener.getListenerCount(BildListener.class) == 0) {
				final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
				if (dav != null) {
					final DynamicObjectType type = (DynamicObjectType) dav.getDataModel().getType(Bild.PID);
					type.removeObjectCreationListener(internalListener);
					type.removeInvalidationListener(internalListener);
					type.removeNameChangeListener(internalListener);
				}
				registered = false;
			}
			if (forced) {
				registered = false;
			}
		}
	}

	/**
	 * Verteilt das Ereignis, dass ein Bild angelegt wurde.
	 *
	 * @param bild
	 *            das neue Bild.
	 */
	protected synchronized void fireBildAngelegt(final Bild bild) {
		final BildEvent e = new BildEvent(this, bild);
		for (final BildListener l : listener.getListeners(BildListener.class)) {
			l.bildAngelegt(e);
		}
	}

	/**
	 * Verteilt das Ereignis, dass ein Bild gelöscht wurde.
	 *
	 * @param bild
	 *            das gelöschte Bild.
	 */
	protected synchronized void fireBildEntfernt(final Bild bild) {
		final BildEvent e = new BildEvent(this, bild);
		bilder.remove(bild);
		for (final BildListener l : listener.getListeners(BildListener.class)) {
			l.bildEntfernt(e);
		}
	}

	/**
	 * Verteilt das Ereignis, dass ein Bild umbenannt wurde.
	 *
	 * @param bild
	 *            das umbenannte Bild.
	 */
	protected synchronized void fireBildUmbenannt(final Bild bild) {
		final BildEvent e = new BildEvent(this, bild);
		for (final BildListener l : listener.getListeners(BildListener.class)) {
			l.bildUmbenannt(e);
		}
	}

	/**
	 * Verteilt das Ereignis, dass sich der Zustand der Verwaltungsverbindung
	 * für Bildobjekte geändert hat.
	 */
	protected synchronized void fireBildConnection() {
		final BildEvent e = new BildEvent(this, null);
		for (final BildListener l : listener.getListeners(BildListener.class)) {
			l.bildConnection(e);
		}
	}

	/**
	 * Liest alle existierenden Bilder aus der Konfiguration.
	 *
	 * @return die aktuell vorhandenen Bilder.
	 */
	public Collection<Bild> getBilder() {
		final ObjektFactory objektFactory = RahmenwerkService.getService().getObjektFactory();
		if (objektFactory.isVerbunden()) {
			for (final SystemObjekt so : objektFactory.bestimmeModellobjekte(Bild.PID)) {
				if (!bilder.containsKey(so)) {
					bilder.put((Bild) so, null);
				}
			}
		}
		return bilder.keySet();
	}

	/**
	 * Speichert ein Bild als dynamisches Objekt in der Konfiguration des
	 * Datenverteilers.
	 *
	 * @param name
	 *            der Name des anzulegenden dynamisches Objekts.
	 * @param pid
	 *            die PID des anzulegenden dynamsiches Objekts
	 * @param mimeType
	 *            der MIME-Type des zu speichernden Bildes.
	 * @param buffer
	 *            die Bilddaten.
	 * @throws DynObjektException
	 */
	public void anlegenBild(final String name, final String pid, final String mimeType, final byte[] buffer)
			throws DynObjektException {

		final ObjektFactory objektFactory = RahmenwerkService.getService().getObjektFactory();
		final KdBild.Daten daten = new KdBild.Daten(new KdBild(null, objektFactory), KdBild.Aspekte.Eigenschaften);
		daten.setMIMEType(mimeType);
		for (final byte element : buffer) {
			daten.getBilddaten().add(new AttByte_JavaKeyword(element));
		}

		objektFactory.createDynamischesObjekt(Bild.class, name, pid, daten);
	}

	/**
	 * Entfernt ein Bild wieder aus der Konfiguration des Datenverteilers.
	 *
	 * @param bild
	 *            da zu löschende Bild.
	 * @throws DynObjektException
	 */
	public void entfernenBild(final Bild bild) throws DynObjektException {
		RahmenwerkService.getService().getObjektFactory().invalidateDynamischesObjekt(bild);
	}

	/**
	 * Gibt für ein Bild den dazugehörigen {@link ImageDescriptor} zurück.
	 *
	 * @param bild
	 *            ein Bild.
	 * @return der entsprechende {@link ImageDescriptor}.
	 */
	public ImageDescriptor getImageDescriptor(final Bild bild) {
		return getLimitedImageDescriptor(bild, 0, 0);
	}

	/**
	 * Gibt für ein Bild den dazugehörigen {@link ImageDescriptor} zurück.
	 *
	 * @param bild
	 *            ein Bild.
	 * @return der entsprechende {@link ImageDescriptor}.
	 */
	public ImageDescriptor getLimitedImageDescriptor(final Bild bild, final int maxWidth, final int maxHeight) {

		ImageData data = bilder.get(bild);
		if (data == null) {
			final Feld<AttByte_JavaKeyword> bilddaten = bild.getKdBild().getDatum().getBilddaten();

			final byte[] buffer = new byte[bilddaten.size()];
			for (int i = 0; i < bilddaten.size(); ++i) {
				buffer[i] = bilddaten.get(i).byteValue();
			}

			final ByteArrayInputStream in = new ByteArrayInputStream(buffer);
			data = new ImageData(in);

			if (bilder.containsKey(bild)) {
				bilder.put(bild, data);
			}
		}

		double scale = 1;

		/* Bild gegebenenfalls skalieren */
		if ((maxWidth > 0) && (maxWidth < data.width)) {
			scale = (double) maxWidth / data.width;
		}
		if ((maxHeight > 0) && (maxHeight < data.height)) {
			scale = Math.min(scale, (double) maxHeight / data.height);
		}

		if (scale < 1) {
			data = data.scaledTo((int) (data.width * scale), (int) (data.height * scale));
		}

		return ImageDescriptor.createFromImageData(data);
	}

	@Override
	public void verbindungHergestellt(final DavVerbindungsEvent event) {
		registerDavListener();
		fireBildConnection();
	}

	@Override
	public void verbindungGetrennt(final DavVerbindungsEvent event) {
		removeDavListener(true);
		fireBildConnection();
	}

	@Override
	public boolean verbindungHalten(final DavVerbindungsEvent event) {
		return false;
	}
}
