/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import de.bsvrz.buv.plugin.dobj.internal.DarstellungExtensionPoint;
import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.dobj.model.ConfiguratedDoTyp;
import de.bsvrz.buv.plugin.dobj.model.DoTyp;
import de.bsvrz.buv.plugin.doeditor.model.DoeditorFactory;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTypReferenz;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmdarstellungsobjekteglobal.objekte.DarstellungsObjektTyp;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Hilfsklasse für den Zugriff auf die im System vorhandenen
 * Darstellungsobjekttypen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class DoTypenUtil {

	private static final Debug LOGGER = Debug.getLogger();

	private DoTypenUtil() {
		// es gibt keine Instanzen der Klasse
	}

	/**
	 * Gibt alle Darstellungsobjekttypen zurück, die zur Verfügung stehen.
	 *
	 * @return alle bekannten Darstellungsobjekttypen.
	 */
	public static Collection<DoTyp> getDoTypen() {
		final Collection<DoTyp> typen = new ArrayList<>();

		// Programmierte DO-Typen
		typen.addAll(DarstellungExtensionPoint.getDoTypen());

		// Editor-DO-Typen
		final ObjektFactory factory = RahmenwerkService.getService().getObjektFactory();
		final List<? extends SystemObjekt> systemObjekte = factory.bestimmeModellobjekte(DarstellungsObjektTyp.PID);
		for (final SystemObjekt so : systemObjekte) {
			try {
				final EditorDoTypReferenz doTyp = DoeditorFactory.eINSTANCE.createEditorDoTypReferenz();
				doTyp.setSystemObjekt(so);
				typen.add(doTyp);
			} catch (final IllegalStateException exception) {
				LOGGER.warning("Der DOT: " + so.getName() + " konnte nicht geladen werden!: "
						+ exception.getLocalizedMessage());
			}
		}

		return typen;
	}

	/**
	 * Gibt alle Darstellungsobjekttypen zurück, die zur für einen bestimmten
	 * Systemobjekttyp Verfügung stehen. Es werden die Darstellungsobjekttypen
	 * zurückgegegeben, die exakt den angegebenen Systemobjekttyp besitzen.
	 *
	 * @param systemObjectType
	 *            der Systemobjekttyp für den die passenden Darstellungsobjekten
	 *            gesucht sind.
	 * @return alle bekannten Darstellungsobjekttypen zum übergebenen
	 *         Systemobjekttyp.
	 */
	public static Collection<DoTyp> getDoTypenWithExactMatch(final SystemObjectType systemObjectType) {
		if (systemObjectType == null) {
			throw new IllegalArgumentException("Argument darf nicht null sein.");
		}

		// Programmierte DO-Typen
		final Collection<DoTyp> filteredTypen = new ArrayList<>();
		filteredTypen.addAll(DarstellungExtensionPoint.getDoTypenWithExactMatch(systemObjectType));

		// Editor-DO-Typen
		final ObjektFactory factory = RahmenwerkService.getService().getObjektFactory();
		final List<? extends SystemObjekt> systemObjekte = factory.bestimmeModellobjekte(DarstellungsObjektTyp.PID);
		for (final SystemObjekt so : systemObjekte) {
			final EditorDoTypReferenz doTyp = DoeditorFactory.eINSTANCE.createEditorDoTypReferenz();
			doTyp.setSystemObjekt(so);
			if (systemObjectType.equals(doTyp.getSystemObjectType())) {
				filteredTypen.add(doTyp);
			}
		}

		return filteredTypen;
	}

	/**
	 * Gibt alle Darstellungsobjekttypen zurück, die zur für einen bestimmten
	 * Systemobjekttyp Verfügung stehen. Es werden die Darstellungsobjekttypen
	 * zurückgegegeben, die angegebenen Systemobjekttyp oder einem seiner
	 * Supertypen besitzen.
	 *
	 * @param systemObjectType
	 *            der Systemobjekttyp für den die passenden Darstellungsobjekten
	 *            gesucht sind.
	 * @return alle bekannten Darstellungsobjekttypen zum übergebenen
	 *         Systemobjekttyp.
	 */
	public static Collection<DoTyp> getDoTypen(final SystemObjectType systemObjectType) {
		if (systemObjectType == null) {
			throw new IllegalArgumentException("Argument darf nicht null sein.");
		}

		// Programmierte DO-Typen
		final Collection<DoTyp> filteredTypen = new ArrayList<>();
		filteredTypen.addAll(DarstellungExtensionPoint.getDoTypen(systemObjectType));

		// Editor-DO-Typen
		final ObjektFactory factory = RahmenwerkService.getService().getObjektFactory();
		final List<? extends SystemObjekt> systemObjekte = factory.bestimmeModellobjekte(DarstellungsObjektTyp.PID);
		for (final SystemObjekt so : systemObjekte) {
			final EditorDoTypReferenz doTyp = DoeditorFactory.eINSTANCE.createEditorDoTypReferenz();
			doTyp.setSystemObjekt(so);
			if (systemObjectType.equals(doTyp.getSystemObjectType())
					|| systemObjectType.getSuperTypes().contains(doTyp.getSystemObjectType())) {
				filteredTypen.add(doTyp);
			}
		}

		return filteredTypen;
	}

	/**
	 * Gibt einen programmierten Darstellungsobjekttyp anhand seiner Extension
	 * ID zurück.
	 *
	 * @param extensionId
	 *            die ID der Extension mit der der Darstellungsobjekttyp
	 *            defniert wurde.
	 * @return der Darstellungsobjekttyp oder <code>null</code>, wenn zu der ID
	 *         keine Extension für einen Darstellungsobjekt existiert.
	 */
	public static ConfiguratedDoTyp getConfiguratedDoTyp(final String extensionId) {
		return DarstellungExtensionPoint.getDoTyp(extensionId);
	}

	/**
	 * Gibt einen Editor-Darstellungsobjekttyp anhand seiner PID zurück.
	 *
	 * @param pid
	 *            die PID des Systemobjekts im Datenverteiler mit der der
	 *            Darstellungsobjekttyp defniert wurde.
	 * @return der Darstellungsobjekttyp oder <code>null</code>, wenn zu der PID
	 *         keine Darstellungsobjekt in der Konfiguration existiert.
	 */
	public static EditorDoTypReferenz getEditorDoTyp(final String pid) {
		final EditorDoTypReferenz doTyp = DoeditorFactory.eINSTANCE.createEditorDoTypReferenz();
		final SystemObjekt so = RahmenwerkService.getService().getObjektFactory().getModellobjekt(pid);
		doTyp.setSystemObjekt(so);
		return doTyp;
	}

}
