/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.buv.plugin.dobj.util;

import java.util.Collection;
import java.util.HashSet;

import javax.swing.event.EventListenerList;

import com.bitctrl.resource.PortableBitmap;

import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsEvent;
import de.bsvrz.buv.rw.basislib.dav.DavVerbindungsListener;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.DynamicObjectType.DynamicObjectCreatedListener;
import de.bsvrz.dav.daf.main.config.DynamicObjectType.NameChangeListener;
import de.bsvrz.dav.daf.main.config.InvalidationListener;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.attribute.AtlPortableBitMap;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.konfigurationsdaten.KdPortableBitMap;
import de.bsvrz.sys.funclib.bitctrl.modell.bitctrlallgemein.objekte.PortableBitMap;
import de.bsvrz.sys.funclib.bitctrl.modell.fachmodellglobal.attribute.AttZahlPositiv;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;

/**
 * Verwaltet dynamische Objekte vom Typ <code>typ.portableBitMap</code>.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public final class PortableBitMapManager implements DavVerbindungsListener {

	/** Die einzige Instanz des Managers. */
	public static final PortableBitMapManager INSTANCE = new PortableBitMapManager();

	private class InternalListener implements DynamicObjectCreatedListener, InvalidationListener, NameChangeListener {

		@Override
		public void invalidObject(final DynamicObject so) {
			firePortableBitMapEntfernt(
					(PortableBitMap) RahmenwerkService.getService().getObjektFactory().getModellobjekt(so));
		}

		@Override
		public void objectCreated(final DynamicObject so) {
			firePortableBitMapAngelegt(
					(PortableBitMap) RahmenwerkService.getService().getObjektFactory().getModellobjekt(so));
		}

		@Override
		public void nameChanged(final DynamicObject so) {
			firePortableBitMapUmbenannt(
					(PortableBitMap) RahmenwerkService.getService().getObjektFactory().getModellobjekt(so));
		}

	}

	private final EventListenerList listener = new EventListenerList();
	private final InternalListener internalListener = new InternalListener();
	private boolean registered;

	private PortableBitMapManager() {
		RahmenwerkService.getService().getRahmenWerk().addDavVerbindungsListener(this);
	}

	/**
	 * Meldet einen Listener auf Änderungen an den vorhandenen
	 * {@link PortableBitMap}s ab.
	 */
	public synchronized void addPortableBitMapListener(final PortableBitMapListener l) {
		listener.add(PortableBitMapListener.class, l);
		registerDavListener();
	}

	private void registerDavListener() {
		if (!registered) {
			if (listener.getListenerCount(PortableBitMapListener.class) == 1) {
				final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
				if (dav != null) {
					final DynamicObjectType type = (DynamicObjectType) dav.getDataModel().getType(PortableBitMap.PID);
					type.addObjectCreationListener(internalListener);
					type.addInvalidationListener(internalListener);
					type.addNameChangeListener(internalListener);
					registered = true;
				}
			}
		}
	}

	/**
	 * Meldet einen Listener auf Änderungen an den vorhandenen
	 * {@link PortableBitMap}s wieder ab.
	 */
	public synchronized void removePortableBitMapListener(final PortableBitMapListener l) {
		listener.remove(PortableBitMapListener.class, l);

		removeDavListener(false);
	}

	private void removeDavListener(final boolean force) {
		if (registered) {
			if (listener.getListenerCount(PortableBitMapListener.class) == 0) {
				final ClientDavInterface dav = RahmenwerkService.getService().getRahmenWerk().getDavVerbindung();
				if (dav != null) {
					final DynamicObjectType type = (DynamicObjectType) dav.getDataModel().getType(PortableBitMap.PID);
					type.removeObjectCreationListener(internalListener);
					type.removeInvalidationListener(internalListener);
					type.removeNameChangeListener(internalListener);
				}
				registered = false;
			}
			if (force) {
				registered = false;
			}
		}
	}

	/**
	 * Verteilt das Ereignis, dass der Zustand der Datenverteilerverbindung zur
	 * Verwaltung der Bitmaps geändert wurde.
	 */
	protected synchronized void firePortableBitMapConnection() {
		final PortableBitMapEvent e = new PortableBitMapEvent(this, null);
		for (final PortableBitMapListener l : listener.getListeners(PortableBitMapListener.class)) {
			l.portableBitMapConnection(e);
		}
	}

	/**
	 * Verteilt das Ereignis, dass eine {@link PortableBitMap} angelegt wurde.
	 *
	 * @param portableBitMap
	 *            die neue Bitmap.
	 */
	protected synchronized void firePortableBitMapAngelegt(final PortableBitMap portableBitMap) {
		final PortableBitMapEvent e = new PortableBitMapEvent(this, portableBitMap);
		for (final PortableBitMapListener l : listener.getListeners(PortableBitMapListener.class)) {
			l.portableBitMapAngelegt(e);
		}
	}

	/**
	 * Verteilt das Ereignis, dass eine {@link PortableBitMap} gelöscht wurde.
	 *
	 * @param portableBitMap
	 *            die gelöschte Bitmap.
	 */
	protected synchronized void firePortableBitMapEntfernt(final PortableBitMap portableBitMap) {
		final PortableBitMapEvent e = new PortableBitMapEvent(this, portableBitMap);
		for (final PortableBitMapListener l : listener.getListeners(PortableBitMapListener.class)) {
			l.portableBitMapEntfernt(e);
		}
	}

	/**
	 * Verteilt das Ereignis, dass eine {@link PortableBitMap} umbenannt wurde.
	 *
	 * @param portableBitMap
	 *            die umbenannte Bitmap.
	 */
	protected synchronized void firePortableBitMapUmbenannt(final PortableBitMap portableBitMap) {
		final PortableBitMapEvent e = new PortableBitMapEvent(this, portableBitMap);
		for (final PortableBitMapListener l : listener.getListeners(PortableBitMapListener.class)) {
			l.portableBitMapUmbenannt(e);
		}
	}

	/**
	 * Liest alle existierenden {@link PortableBitMap}s aus der Konfiguration.
	 *
	 * @return die aktuell vorhandenen {@link PortableBitMap}s.
	 */
	public Collection<PortableBitMap> getPortableBitMaps() {
		final Collection<PortableBitMap> bitmaps = new HashSet<>();
		final ObjektFactory objektFactory = RahmenwerkService.getService().getObjektFactory();
		if (objektFactory.isVerbunden()) {
			for (final SystemObjekt so : objektFactory.bestimmeModellobjekte(PortableBitMap.PID)) {
				bitmaps.add((PortableBitMap) so);
			}
		}
		return bitmaps;
	}

	/**
	 * Speichert eine {@link PortableBitMap} als dynamisches Objekt in der
	 * Konfiguration des Datenverteilers.
	 *
	 * @param name
	 *            der Name des anzulegenden dynamisches Objekts.
	 * @param pid
	 *            die PID des anzulegenden dynamsiches Objekts
	 * @param mimeType
	 *            der MIME-Type des zu speichernden Bildes.
	 * @param buffer
	 *            die Bilddaten.
	 * @throws DynObjektException
	 */
	public void anlegenPortableBitMap(final String name, final String pid, final PortableBitmap bitmap)
			throws DynObjektException {
		final ObjektFactory objektFactory = RahmenwerkService.getService().getObjektFactory();
		final KdPortableBitMap.Daten daten = konvertieren(bitmap);
		objektFactory.createDynamischesObjekt(PortableBitMap.class, name, pid, daten);
	}

	private KdPortableBitMap.Daten konvertieren(final PortableBitmap bitmap) {

		final KdPortableBitMap.Daten daten = new KdPortableBitMap.Daten(
				new KdPortableBitMap(null, RahmenwerkService.getService().getObjektFactory()),
				KdPortableBitMap.Aspekte.Eigenschaften);
		final AtlPortableBitMap bildDaten = new AtlPortableBitMap();

		bildDaten.getAusmasse().setBreite(new AttZahlPositiv((long) bitmap.getWidth()));
		bildDaten.getAusmasse().setHoehe(new AttZahlPositiv((long) bitmap.getHeight()));

		final int maxLen = 70;
		final StringBuffer buffer = new StringBuffer();
		for (int zeile = 0; zeile < bitmap.getHeight(); zeile++) {
			for (int spalte = 0; spalte < bitmap.getWidth(); spalte++) {
				if ((buffer.length() >= maxLen) || (buffer.length() >= bitmap.getWidth())) {
					bildDaten.getMaske().add(buffer.toString());
					buffer.setLength(0);
				}
				if (bitmap.getPixel(spalte, zeile) == 0) {
					buffer.append('0');
				} else {
					buffer.append('1');
				}
			}
		}
		if (buffer.length() > 0) {
			bildDaten.getMaske().add(buffer.toString());
		}
		daten.setBilddaten(bildDaten);
		return daten;
	}

	/**
	 * Entfernt eine {@link PortableBitMap} wieder aus der Konfiguration des
	 * Datenverteilers.
	 *
	 * @param portableBitMap
	 *            die zu entfernende Bitmap.
	 * @throws DynObjektException
	 */
	public void entfernenPortableBitMap(final PortableBitMap portableBitMap) throws DynObjektException {
		RahmenwerkService.getService().getObjektFactory().invalidateDynamischesObjekt(portableBitMap);
	}

	public void aktualisierenPortableBitMap(final PortableBitMap bitmap, final PortableBitmap bildDaten)
			throws ConfigurationChangeException {

		final KdPortableBitMap.Daten daten = konvertieren(bildDaten);
		bitmap.getKdPortableBitMap().setDatum(daten);
	}

	@Override
	public void verbindungHergestellt(final DavVerbindungsEvent event) {
		registerDavListener();
		firePortableBitMapConnection();
	}

	@Override
	public void verbindungGetrennt(final DavVerbindungsEvent event) {
		removeDavListener(true);
		firePortableBitMapConnection();
	}

	@Override
	public boolean verbindungHalten(final DavVerbindungsEvent event) {
		return false;
	}
}
