/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.vektor;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.draw2d.Ellipse;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.Polygon;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;

/**
 * Hilfsmethoden fuer linear-algebraische Operationen.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
public final class LinAlgUtil {

	private LinAlgUtil() {
		// es gibt keine Instanzen der Klasse
	}

	private static boolean isRedundant(final ParallelKomplex pkfMinus1, final ParallelKomplex pkf,
			final ParallelKomplex pkfPlus1) {
		final Point pVor = pkf.calcSchnittPunktMitVorgaenger(pkfMinus1);
		final Point pNach = pkf.calcSchnittPunktMitNachfolger(pkfPlus1);
		final Vektor2D vStrich = new Vektor2D(pVor, pNach);
		final double winkel = getWinkel(pkf.getV(), vStrich);
		return winkel > 3.14;
	}

	/**
	 * TODO: Wichtungen richtig einstellen!
	 *
	 * @param originalPunkte
	 * @param abstand
	 * @param scale
	 * @return
	 */

	public static PointList verschiebeUndStrecke(final PointList originalPunkte, final int abstand,
			final double scale) {
		// TODO Über auskommentierten Quelltext nachdenken, ggf. löschen
		// final int mindestAbstand;
		// final double abstandAbsolut = scale * abstand;
		// if (abstandAbsolut > 5.0) {
		// if (scale < 0.005) {
		// mindestAbstand = 1000;
		// } else if (scale < 0.001) {
		// mindestAbstand = 2500;
		// } else {
		// return verschiebe(originalPunkte, abstand);
		// }
		// return verschiebeNeu(strecke(originalPunkte, mindestAbstand),
		// abstand);
		// }
		return verschiebe(originalPunkte, abstand);
	}

	/**
	 * Verschiebt eine Menge von Punkten (ein Polygon) quasiparallel um eine
	 * bestimmte Menge Pixel.
	 *
	 * @param originalPunkte
	 *            die Menge der Punkte, die das zu verschiebende Polygon
	 *            beschreibt.
	 * @param abstand
	 *            der Abstand in Pixel, um den das Polygon verschoben werden
	 *            soll. Positiv heisst im Uhrzeigersinn, negativ dagegen.
	 * @return eine Punktmenge, die das verschobene Polygon beschreibt.
	 */
	public static PointList verschiebe(final PointList originalPunkte, final int abstand) {
		if (abstand == 0) {
			return originalPunkte;
		}

		final List<LineareFunktion> funktionen = getLineareFunktionenParallelZumPolygon(originalPunkte, abstand);

		if (funktionen.isEmpty()) {
			return new PointList();
		}

		final PointList bildPunkte = new PointList();
		bildPunkte.addPoint(funktionen.get(0).getStartPunkt());

		for (int i = 0; i < funktionen.size(); i++) {
			final LineareFunktion fi = funktionen.get(i);

			/**
			 * Noch nicht das letzte Teilstueck?
			 */
			if (i < funktionen.size() - 1) {
				final LineareFunktion fiPlus1 = funktionen.get(i + 1);

				try {
					Point schnittPunkt = getSchnittpunkt(fi, fiPlus1);

					if (schnittPunkt == null) {
						return new PointList();
					}
					/**
					 * Workaround
					 */
					if (schnittPunkt.getDistance(fi.getEndPunkt()) > abstand) {
						schnittPunkt = fi.getEndPunkt();
					}
					bildPunkte.addPoint(schnittPunkt);
				} catch (final IllegalArgumentException ex) {
					// NetzPlugin.getDefault().getLogger().warning(
					// ex.getLocalizedMessage());
					return new PointList();
				}
			} else {
				bildPunkte.addPoint(fi.getEndPunkt());
			}
		}

		return bildPunkte;
	}

	/**
	 * Verschiebt eine Menge von Punkten (ein Polygon) quasiparallel um eine
	 * bestimmte Menge Pixel.
	 *
	 * @param originalPunkte
	 *            die Menge der Punkte, die das zu verschiebende Polygon
	 *            beschreibt.
	 * @param abstand
	 *            der Abstand in Pixel, um den das Polygon verschoben werden
	 *            soll. Positiv heisst im Uhrzeigersinn, negativ dagegen.
	 * @return eine Punktmenge, die das verschobene Polygon beschreibt.
	 */
	public static PointList verschiebeNeu(final PointList originalPunkte, final int abstand) {
		final PointList verschiedenePunkte = getVerschiedenePunkte(originalPunkte);
		if (verschiedenePunkte.size() <= 1) {
			return originalPunkte;
		}

		final PKStack stack = new PKStack();
		final List<ParallelKomplex> komplexe = new ArrayList<>();

		for (int i = 0; i < verschiedenePunkte.size() - 1; i++) {
			final Point p1 = verschiedenePunkte.getPoint(i);
			final Point p2 = verschiedenePunkte.getPoint(i + 1);
			komplexe.add(new ParallelKomplex(p1, p2, abstand));
		}

		for (int i = 0; i < komplexe.size(); i++) {
			final ParallelKomplex pkfMinus1 = stack.popGet();
			final ParallelKomplex pkf = komplexe.get(i);
			final ParallelKomplex pkfPlus1;
			if (i + 1 == komplexe.size()) {
				pkfPlus1 = null;
			} else {
				pkfPlus1 = komplexe.get(i + 1);
			}
			final boolean redundant = isRedundant(pkfMinus1, pkf, pkfPlus1);
			if (!redundant) {
				stack.push(pkf);
			}
		}

		return stack.getPunkte();
	}

	/**
	 * Erfragt den Winkel zwischen zwei Vektoren a und b.
	 *
	 * @param a
	 *            Vektor a.
	 * @param b
	 *            Vektor b.
	 * @return der Winkel zwischen a und b.
	 */
	public static double getWinkel(final Vektor2D a, final Vektor2D b) {
		final double d = a.getLaenge() * b.getLaenge();
		if (d == 0.0) {
			return 0.0;
		}
		return Math.acos((a.x * b.x + a.y * b.y) / d);
	}

	/**
	 * Ermittelt eine Liste von linearen Funktionen, die jeweils parallel zu den
	 * uebergebenen Teilstrecken eines Polygons verlaufen. Die Funktionen haben
	 * jeweils einen bestimmten Abstand zur assoziierten Polygonteilstueck.
	 *
	 * @param punkte
	 *            Liste der Punkte eines Polygons.
	 * @param abstand
	 *            der Abstand in Pixel von der Funktion zum Polygonteilstueck.
	 *            Positiv heisst im Uhrzeigersinn, negativ dagegen.
	 * @return eine ggf. leere Liste von linearen Funktionen (leer, wenn in der
	 *         Punktmenge weniger als zwei unterschiedliche Punkte liegen).
	 */
	public static List<LineareFunktion> getLineareFunktionenParallelZumPolygon(final PointList punkte,
			final double abstand) {
		final PointList verschiedenePunkte = getVerschiedenePunkte(punkte);
		final List<LineareFunktion> funktionen = new ArrayList<>();

		if (verschiedenePunkte.size() > 1) {
			final boolean rechtenNormalenVektor = abstand > 0;
			final double derAbstand = Math.abs(abstand);
			for (int i = 0; i < verschiedenePunkte.size() - 1; i++) {
				final Point a = verschiedenePunkte.getPoint(i);
				final Point b = verschiedenePunkte.getPoint(i + 1);

				final Vektor2D normalenVektorNormiert;
				final Vektor2D vab = new Vektor2D(a, b);
				if (rechtenNormalenVektor) {
					normalenVektorNormiert = vab.getNormalenVektorRechts().getVektorNormiert();
				} else {
					normalenVektorNormiert = vab.getNormalenVektorLinks().getVektorNormiert();
				}

				funktionen.add(new LineareFunktion(
						new PrecisionPoint(a.x + derAbstand * normalenVektorNormiert.x,
								a.y + derAbstand * normalenVektorNormiert.y),
						new PrecisionPoint(b.x + derAbstand * normalenVektorNormiert.x,
								b.y + derAbstand * normalenVektorNormiert.y)));
			}
		}

		return funktionen;
	}

	/**
	 * Ermittelt aus einer Liste von Punkten eine Liste <code>pl</code> von
	 * Punkten fuer die gilt: <code>pl(x) != pl (x + 1)</code>
	 *
	 * @param punkte
	 *            eine Liste von Punkten in der Ebene.
	 * @return eine Liste <code>pl</code> von Punkten fuer die gilt:
	 *         <code>pl(x) != pl (x + 1)</code>.
	 */
	private static PointList getVerschiedenePunkte(final PointList punkte) {
		if (punkte.size() == 0) {
			return new PointList();
		}
		final PointList verschiedenePunkte = new PointList();
		verschiedenePunkte.addPoint(punkte.getFirstPoint());
		if (punkte.size() > 1) {
			for (int i = 1; i < punkte.size(); i++) {
				final Point naechsterPunkt = punkte.getPoint(i);
				if (!naechsterPunkt.equals(verschiedenePunkte.getLastPoint())) {
					verschiedenePunkte.addPoint(naechsterPunkt);
				}
			}
		}

		return verschiedenePunkte;
	}

	/**
	 * Streckt eine Menge von Punkten (ein Polygon).<br>
	 * Die Streckung des Polygons erfolgt durch die Ersetzung aller Punkte,
	 * deren Abstand untereinander geringer als <code>mindestAbstandy</code> ist
	 * durch deren Schwerpunkt.
	 *
	 * @param punkte
	 *            eine Menge von Punkten.
	 * @param mindestAbstand
	 *            der Mindestabstand zwischen zwei Punkten innerhalb des
	 *            Polygons.
	 * @return die Punkte des gesreckten Poygons.
	 */
	public static PointList strecke(final PointList punkte, final double mindestAbstand) {
		if (punkte == null) {
			return new PointList();
		}
		if (punkte.size() > 2) {
			final PointList abbildung = new PointList();
			PointList streckListe = new PointList();
			abbildung.addPoint(punkte.getFirstPoint());
			for (int i = 0; i < punkte.size() - 1; i++) {
				final Point p1 = punkte.getPoint(i);
				final Point p2 = punkte.getPoint(i + 1);
				final Vektor2D v = new Vektor2D(p1, p2);
				if (v.getLaenge() < mindestAbstand) {
					streckListe.addPoint(p2);
				} else {
					if (streckListe.size() > 1) {
						abbildung.addPoint(getSchwerpunkt(streckListe));
					}
					abbildung.addPoint(p2);
					streckListe = new PointList();
				}
			}
			if (streckListe.size() > 1) {
				streckListe.removePoint(streckListe.size() - 1);
				abbildung.addPoint(getSchwerpunkt(streckListe));
			}
			abbildung.addPoint(punkte.getLastPoint());

			return abbildung;
		}
		return punkte;
	}

	/**
	 * Berechnet den "Schwerpunkt" einer Menge von Punkten (geometrischer
	 * Mittelwert).
	 *
	 * @param punktMenge
	 *            eine Punktemenge.
	 * @return den "Schwerpunkt".
	 */
	private static Point getSchwerpunkt(final PointList punktMenge) {
		double x = 0.0;
		double y = 0.0;
		for (int i = 0; i < punktMenge.size(); i++) {
			final Point p = punktMenge.getPoint(i);
			x += p.preciseX();
			y += p.preciseY();
		}
		return new PrecisionPoint(x / (punktMenge.size()), y / (punktMenge.size()));
	}

	/**
	 * Errechnet einen Punkt, der in einem bestimmten Abstand rechts oder links
	 * neben dem Anfang (-spunkt) der Linie liegt.
	 *
	 * @param punkte
	 *            die Linie.
	 * @param abstand
	 *            der Abstand (im rechten Winkel) von der Linie (positiv heisst
	 *            im Uhrzeigersinn, negativ dagegen).
	 * @return ein Punkt, der in einem bestimmten Abstand rechts oder links
	 *         neben dem Anfangspunkt der Linie liegt oder <code>null</code>,
	 *         wenn dieser Punkt nicht berechnet werden konnte.
	 */
	public static Point getPunktNebenPunktListeAnfang(final PointList punkte, final double abstand) {
		final Vektor2D normalenVektor2d = getNormalenVektorZumErstenLinienZugVon(punkte, abstand >= 0);
		if (normalenVektor2d != null) {
			final Point p1 = punkte.getFirstPoint();
			final double absAbstand = Math.abs(abstand);
			return new PrecisionPoint(p1.x + normalenVektor2d.getVektorNormiert().x * absAbstand,
					p1.y + normalenVektor2d.getVektorNormiert().y * absAbstand);
		}
		return null;
	}

	/**
	 * Errechnet den Normalenvektor zum ersten Linienzug dieses Polygons.
	 *
	 * @param punkte
	 *            die Punkte des Polygons.
	 * @param rechts
	 *            rechter Normalenvektor?
	 *
	 * @return der Normalenvektor oder <code>null</code>, wenn dieser Punkt
	 *         nicht berechnet werden konnte.
	 */
	public static Vektor2D getNormalenVektorZumErstenLinienZugVon(final PointList punkte, final boolean rechts) {
		if (punkte.size() > 1) {
			final Point p1 = punkte.getFirstPoint();
			Point p2;
			p2 = punkte.getPoint(1);
			for (int i = 2; p1.equals(p2) && i < punkte.size(); i++) {
				p2 = punkte.getPoint(i);
			}
			if (!p1.equals(p2)) {
				final Vektor2D v = new Vektor2D(p1, p2);
				if (rechts) {
					return v.getNormalenVektorRechts();
				}
				return v.getNormalenVektorLinks();
			}
		}
		return null;
	}

	/**
	 * Ermittelt den Schnittpunkt zwischen zwei linearen Funktionen
	 * <code>f1</code> und <code>f2</code>.
	 *
	 * @param f1
	 *            eine lineare Funktion im Kartenkoordinatensystem.
	 * @param f2
	 *            eine lineare Funktion im Kartenkoordinatensystem.
	 * @return der Schnittpunkt der Funktionen oder <code>null</code>, wenn die
	 *         Funktionen parallel verlaufen.
	 */
	public static Point getSchnittpunkt(final LineareFunktion f1, final LineareFunktion f2) {
		double xSchnittPunkt;
		double ySchnittPunkt;
		xSchnittPunkt = 0;
		ySchnittPunkt = 0;

		if (f1.isYParallele() || f2.isYParallele()) {
			if (f1.isYParallele() && f2.isYParallele()) {
				if (f1.getStartPunkt().x == f2.getStartPunkt().x) {
					return f1.getEndPunkt();
				}
				/**
				 * Beide Funktionen liegen parallel nebeneinander (parallel zur
				 * Y-Achse)
				 */
				return null;
			} else if (f1.isYParallele()) {
				xSchnittPunkt = f1.getStartPunkt().x;
				ySchnittPunkt = f2.getFunktionsWert(xSchnittPunkt);
			} else {
				xSchnittPunkt = f2.getStartPunkt().x;
				ySchnittPunkt = f1.getFunktionsWert(xSchnittPunkt);
			}
		} else {
			final double m1MinusM2 = f1.getAnstieg() - f2.getAnstieg();

			if (m1MinusM2 == 0) {
				if (f1.getAbsolutglied() != f2.getAbsolutglied()) {
					/**
					 * Beide Funktionen liegen parallel nebeneinander.
					 */
					return null;
				}
				return f1.getEndPunkt();
			}

			xSchnittPunkt = (f2.getAbsolutglied() - f1.getAbsolutglied()) / m1MinusM2;
			ySchnittPunkt = f1.getAnstieg() * xSchnittPunkt + f1.getAbsolutglied();
		}
		final Point schnittpunkt = new PrecisionPoint(xSchnittPunkt, ySchnittPunkt);

		return schnittpunkt;
	}

	/**
	 * Erfragt den Schnittpunkt zwischen zwei Strecken s1 und s2.
	 *
	 * @param s1p1
	 *            Startpunkt der Strecke Nr.1.
	 * @param s1p2
	 *            Endpunkt der Strecke Nr.1.
	 * @param s2p1
	 *            Startpunkt der Strecke Nr.2.
	 * @param s2p2
	 *            Endpunkt der Strecke Nr.2.
	 * @return der Schnittpunkt zwischen s1 und s2 oder <code>null</code>, wenn
	 *         sich die Strecken nicht schneiden.
	 */
	public static Point getSchnittpunkt(final Point s1p1, final Point s1p2, final Point s2p1, final Point s2p2) {
		final LineareFunktion f1 = new LineareFunktion(s1p1, s1p2);
		final LineareFunktion f2 = new LineareFunktion(s2p1, s2p2);

		final Point schnittPunkt = getSchnittpunkt(f1, f2);
		if (schnittPunkt != null) {
			final Rectangle rs1 = new Rectangle(s1p1, s1p2);
			final Rectangle rs2 = new Rectangle(s2p1, s2p2);
			if (rs1.contains(schnittPunkt) && rs2.contains(schnittPunkt)) {
				return schnittPunkt;
			}
		}
		return null;
	}

	/**
	 * Erfragt, ob eine {@link IFigure} von einem {@link Polygon} umschlossen
	 * ist.
	 *
	 * @param polygon
	 *            das {@link Polygon}.
	 * @param figure
	 *            die {@link IFigure}.
	 * @param intersectionModeOn
	 *            wenn dieses Flag an ist, dann gilt die {@link IFigure} dann
	 *            umschlossen, wenn <b>mindestens ein</b> Eckpunkt vollstaendig
	 *            im {@link Polygon} liegt. Sonst muessen <b>alle</b> Eckpunkte
	 *            im {@link Polygon} liegen.
	 * @return ob die uebergebene {@link IFigure} vom uebergebenen
	 *         {@link Polygon} umschlossen ist.
	 */
	public static boolean polygonContainsRectangle(final PointList polygon, final IFigure figure,
			final boolean intersectionModeOn) {
		int boundsIncluded = 0;
		final Point[] shapeDescriptor = getShapeDescriptor(figure);
		if (polygon.size() > 3) {
			for (final Point boundsPoint : shapeDescriptor) {
				final Point s1p1 = new Point(0, boundsPoint.y);
				final Point s1p2 = new Point(boundsPoint.x, boundsPoint.y);

				int intersectionCount = 0;
				if (!s1p1.equals(s1p2)) {
					for (int i = 1; i < polygon.size(); i++) {
						final Point schnittPunkt = LinAlgUtil.getSchnittpunkt(s1p1, s1p2, polygon.getPoint(i - 1),
								polygon.getPoint(i));
						if (schnittPunkt != null) {
							intersectionCount += 1;
						}
					}
				}
				boundsIncluded += intersectionCount & 1;

				if (intersectionModeOn && boundsIncluded > 0) {
					boundsIncluded = shapeDescriptor.length;
					break;
				}
			}
		}

		return boundsIncluded == shapeDescriptor.length;
	}

	/**
	 * Erfragt die diese {@link IFigure} bestimmenden Grenzpunkte.
	 *
	 * @param figure
	 *            eine ({@link IFigure}).
	 * @return die diese {@link IFigure} bestimmenden Grenzpunkte.
	 */
	private static Point[] getShapeDescriptor(final IFigure figure) {
		if (figure instanceof Ellipse) {
			return new Point[] {
					new Point((figure.getBounds().getTopLeft().x + figure.getBounds().getTopRight().x) / 2,
							figure.getBounds().getTopLeft().y),
					new Point((figure.getBounds().getBottomLeft().x + figure.getBounds().getBottomRight().x) / 2,
							figure.getBounds().getBottomLeft().y),
					new Point(figure.getBounds().getTopLeft().x,
							(figure.getBounds().getTopLeft().y + figure.getBounds().getBottomLeft().y) / 2),
					new Point(figure.getBounds().getTopRight().x,
							(figure.getBounds().getTopRight().y + figure.getBounds().getBottomRight().y) / 2) };
		}
		return new Point[] { figure.getBounds().getTopLeft(), figure.getBounds().getBottomLeft(),
				figure.getBounds().getBottomRight(), figure.getBounds().getTopRight() };
	}

	/**
	 * Erfragt die Menge <b>aller</b> Kinder (auch Kindeskinder usw.) eines
	 * {@link EditPart}.
	 *
	 * @param editPart
	 *            der {@link EditPart}.
	 * @return die ggf. leere Menge aller Kinder.
	 */
	public static Set<EditPart> getAllChildrenOf(final EditPart editPart) {
		final Set<EditPart> allChildren = new HashSet<>();
		if (editPart != null) {
			getAllChildren(editPart, allChildren);
		}
		return allChildren;
	}

	private static void getAllChildren(final EditPart editPart, final Set<EditPart> allChildren) {
		final List<?> children = editPart.getChildren();
		for (int i = 0; i < children.size(); i++) {
			final EditPart child = (EditPart) children.get(i);
			allChildren.add(child);
			getAllChildren(child, allChildren);
		}
	}

}
