/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.dobj.vektor;

import org.eclipse.draw2d.geometry.Point;

/**
 * Lineare Funktion mit zwei Punkten aus dem Wertebereich.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
public class LineareFunktion {

	/**
	 * Anstieg.
	 */
	private final double m;

	/**
	 * Absolutglied.
	 */
	private final double n;

	/**
	 * Ein Punkt, der die Funktion erfuellt.
	 */
	private final Point a;

	/**
	 * Noch ein Punkt, der die Funktion erfuellt.
	 */
	private final Point b;

	/**
	 * Indiziert, ob es sich hier um eine y-Parallele handelt.
	 */
	private final boolean yParallele;

	/**
	 * Konstruktor fuer die Definition der Funktion ueber zwei Punkte die sie
	 * erfuellen.
	 *
	 * @param a
	 *            Ein Punkt, der die Funktion erfuellt.
	 * @param b
	 *            Noch ein Punkt, der die Funktion erfuellt.
	 */
	public LineareFunktion(final Point a, final Point b) {
		if (a.equals(b)) {
			throw new IllegalArgumentException("Punkt a = " + a + " und Punkt b = " + b + " sind identisch.");
		}

		final double xDif = a.preciseX() - b.preciseX();
		if (xDif == 0.0) {
			yParallele = true;
			m = (a.preciseY() - b.preciseY()) > 0.0 ? Double.POSITIVE_INFINITY : Double.NEGATIVE_INFINITY;
			n = Double.NaN;
		} else {
			yParallele = false;
			m = (a.preciseY() - b.preciseY()) / xDif;
			n = a.preciseY() - m * a.preciseX();
		}

		this.a = a;
		this.b = b;
	}

	/**
	 * Berechnet den Funktionswert an der uebergebenen Stelle.
	 *
	 * @param x
	 *            ein Wert aus dem Definitionsbereich.
	 * @return der Funktionswert an der uebergebenen Stelle.
	 */
	public final double getFunktionsWert(final double x) {
		return m * x + n;
	}

	/**
	 * Erfragt Anstieg.
	 *
	 * @return der Anstieg.
	 */
	public final double getAnstieg() {
		return m;
	}

	/**
	 * Erfragt Absolutglied.
	 *
	 * @return das Absolutglied.
	 */
	public final double getAbsolutglied() {
		return n;
	}

	/**
	 * Erfragt, ob es sich hier um eine y-Parallele handelt.
	 *
	 * @return ob es sich hier um eine y-Parallele handelt.
	 */
	public boolean isYParallele() {
		return yParallele;
	}

	/**
	 * Erfragt den Startpunkt, ueber den die Funktion definiert wurde.
	 *
	 * @return der Startpunkt, ueber den die Funktion definiert wurde.
	 */
	public Point getStartPunkt() {
		return a;
	}

	/**
	 * Erfragt den Endpunkt, ueber den die Funktion definiert wurde.
	 *
	 * @return der Endpunkt, ueber den die Funktion definiert wurde.
	 */
	public Point getEndPunkt() {
		return b;
	}

	@Override
	public boolean equals(final Object obj) {
		if (obj instanceof LineareFunktion) {
			final LineareFunktion that = (LineareFunktion) obj;
			if (yParallele || that.yParallele) {
				if (yParallele == that.yParallele) {
					return a.equals(that.a) && b.equals(that.b);
				}
				return false;
			}
			return Double.compare(m, that.m) == 0 && Double.compare(n, that.n) == 0;
		}
		return false;
	}

	@Override
	public String toString() {
		return "f(x) := "
				+ (yParallele ? "Y-Parallele in x = " + a.x : (m + "x " + (n >= 0 ? "+" : "-") + " " + Math.abs(n)))
				+ " (von " + a + " nach " + b + ")";
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((a == null) ? 0 : a.hashCode());
		result = prime * result + ((b == null) ? 0 : b.hashCode());
		long temp;
		temp = Double.doubleToLongBits(m);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(n);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		result = prime * result + (yParallele ? 1231 : 1237);
		return result;
	}

}
