/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.figures;

import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.ScalableFigure;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;

import de.bsvrz.buv.plugin.dobj.DoFigure;
import de.bsvrz.buv.plugin.dobj.decorator.DrehungFigure;
import de.bsvrz.buv.plugin.doeditor.util.DoEditorUtil;

/**
 * Figur, die ein DO-Editor-Objekt in einer Darstellung repräsentiert.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class EditorDoModelFigure extends Figure implements DoFigure, DrehungFigure, ScalableFigure {

	/** die aktuelle Skalierung. */
	private double scale = 1.0;
	/** der aktuelle Drehwinkel. */
	private float winkel;
	/** die Größe des Objekts. */
	private Dimension size;
	/** der lokale Offset für die Anzeige der Figur. */
	private Point locOffset = new Point();
	/** die Position für die Platzierung der Figur. */
	private Point hotspot = new Point();

	/** minimale Zoomstufe bis zu der die Figur angezeigt wird. */
	private double vonZoom;
	/** maximale Zoomstufe bis zu der die Figur angezeigt wird. */
	private double bisZoom = Double.MAX_VALUE;

	/** die aktuelle Zoomstufe. */
	private double zoom = 1;

	@Override
	protected boolean useLocalCoordinates() {
		return true;
	}

	@Override
	public double getScale() {
		return scale;
	}

	@Override
	public void setScale(final double scale) {
		this.scale = scale;
		updateBounds();
		// setSize((int) (size.width * scale), (int) (size.height * scale));
		// setLocation(new Point((int) (hotspot.x / scale),
		// (int) (hotspot.y / scale)));

	}

	@Override
	public float getDrehwinkel() {
		return winkel;
	}

	@Override
	public void setDrehwinkel(final float drehwinkel) {
		winkel = drehwinkel;
		updateBounds();
	}

	@Override
	public Point getHotspot() {
		return hotspot;
	}

	@Override
	public void setHotspot(final Point hotspot) {
		this.hotspot = hotspot;
		updateBounds();
	}

	/** aktualisiert die Abmessungen der Figur. */
	private void updateBounds() {

		final Point calculatedLocation = hotspot.getCopy().translate(locOffset.getNegated()).scale(1 / scale);
		setLocation(calculatedLocation);
		final Rectangle originalBounds = new Rectangle(calculatedLocation, size);
		final Point drehPunkt = hotspot.getCopy().translate(locOffset.getNegated());
		final Rectangle rotatedBounds = DoEditorUtil.getRotatedBounds(originalBounds.scale(scale), winkel, drehPunkt);
		setBounds(rotatedBounds);
	}

	@Override
	public void handleZoomChanged(final double newZoom) {
		zoom = newZoom;
		// setScale(newZoom);
		updateVisibility();
	}

	@Override
	public void setSichtbareZoomStufe(final double newVonZoom, final double newBisZoom) {
		if (vonZoom == newVonZoom && bisZoom == newBisZoom) {
			return;
		}

		vonZoom = newVonZoom;
		bisZoom = newBisZoom;
		updateVisibility();
	}

	/** aktualisiert die Sichtbarkeit der Figur. */
	private void updateVisibility() {
		if (zoom >= vonZoom && zoom <= bisZoom) {
			setVisible(true);
		} else {
			setVisible(false);
		}
	}

	@Override
	public void paint(final Graphics graphics) {

		graphics.pushState();

		final Rectangle rotatedBounds = getBounds().getCopy();
		final Rectangle clipRect = new Rectangle(hotspot, size).getUnion(rotatedBounds);
		graphics.setClip(clipRect.intersect(graphics.getClip(new Rectangle())));

		if (winkel != 0) {
			final Point rotationPt = hotspot.getCopy().translate(locOffset.getNegated());

			graphics.translate(rotationPt);
			graphics.rotate(winkel);
			graphics.translate(rotationPt.getNegated());
			graphics.translate(hotspot.x - getBounds().x, hotspot.y - getBounds().y);
		}

		if (scale != 1) {
			graphics.translate(getLocation());
			graphics.scale(scale);
			graphics.translate(getLocation().getNegated());
		}

		super.paint(graphics);
		graphics.popState();
	}

	/**
	 * setzt die Größe der Figur.
	 *
	 * @param newSize
	 *            die Größe
	 */
	public void setDimension(final Dimension newSize) {
		size = newSize;
		updateBounds();
	}

	/**
	 * setzt den lokalen Offset für die Darstellung der Figur.
	 *
	 * @param locOffset
	 *            der Offset
	 */
	public void setLocOffset(final Point locOffset) {
		if (locOffset == null) {
			this.locOffset = new Point();
		} else {
			this.locOffset = locOffset;
		}
		updateBounds();
	}
}
