/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.figures;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.RectangleFigure;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.graphics.Pattern;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.themes.ColorUtil;

import com.bitctrl.lib.eclipse.emf.eclipse.model.EColor;
import com.bitctrl.lib.eclipse.emf.eclipse.model.ELineAttributes;

import de.bsvrz.buv.plugin.doeditor.DoGraphics;
import de.bsvrz.buv.plugin.doeditor.editparts.DecoratorDatenSatz;
import de.bsvrz.buv.plugin.doeditor.model.BackgroundParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.DrehwinkelParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ForegroundParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.LineAttributeParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.RechteckForm;
import de.bsvrz.buv.plugin.doeditor.model.SichtbarkeitParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhalten;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhaltenParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.util.DoEditorUtil;
import de.bsvrz.buv.plugin.doeditor.util.PatternRegistry;

/**
 * Figur zur Darstellung eines Rechteck-Elements innerhalb eines
 * Dartellungsobjekttyps.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class RechteckFormFigure extends RectangleFigure implements VisibleFormFigure {

	/** das Modellobjekt, das die Rechteckform repräsentiert. */
	private final RechteckForm rechteck;
	/** der Ressourcemanager für verwendete Farben. */
	private final ResourceManager resourceManager;
	/** der Ressourcemanager für verwendete Füllmuster. */
	private final PatternRegistry patterns;

	/** das aktuelle Zoomverhalten. */
	private ZoomVerhalten zoomVerhalten;

	/** die aktuelle Skalierung. */
	private double scale = 1.0;
	/** der aktuelle Drehwinkel. */
	private double currentAngle;

	/**
	 * Konstruktor.
	 *
	 * @param viewer
	 *            der Viewer, in dem die Figur dargestellt werden soll.
	 * @param rechteck
	 *            das Modellobjekt
	 */
	public RechteckFormFigure(final EditPartViewer viewer, final RechteckForm rechteck) {
		resourceManager = viewer.getResourceManager();
		patterns = PatternRegistry.getRegistry(viewer);
		this.rechteck = rechteck;
		zoomVerhalten = rechteck.getZoomVerhalten();
		currentAngle = rechteck.getAngle();
	}

	@Override
	public void aktualisiereVomModel() {

		zoomVerhalten = rechteck.getZoomVerhalten();

		setLocation(rechteck.getLocation());
		currentAngle = rechteck.getAngle();
		updateSize();

		setForegroundColor(resourceManager.createColor(rechteck.getForegroundColor().getRgb()));
		setVisible(rechteck.isVisible());

		// XXX Workaround weil getLineStyle die LineAttributes überschreibt
		// getFigure().setLineAttributes(getModel().getLineAttributes());
		final ELineAttributes lineAttributes = rechteck.getLineAttributes();
		setLineWidthFloat(lineAttributes.getWidth());
		setLineStyle(lineAttributes.getStyle());
		setBackgroundColor(resourceManager.createColor(rechteck.getBackgroundColor().getRgb()));
		setFill(rechteck.isFilled());
	}

	@Override
	public void decorate(final ParameterDefinition decorator, final DecoratorDatenSatz datenSatz) {
		if (decorator instanceof BackgroundParameterDefinition) {
			final EColor startColor = ((BackgroundParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((BackgroundParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				RGB rgb = startColor.getRgb();
				if (endColor != null) {
					rgb = ColorUtil.blend(rgb, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
				setBackgroundColor(resourceManager.createColor(rgb));
			}
		} else if (decorator instanceof ForegroundParameterDefinition) {
			final EColor startColor = ((ForegroundParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((ForegroundParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				RGB rgb = startColor.getRgb();
				if (endColor != null) {
					rgb = ColorUtil.blend(rgb, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
				setForegroundColor(resourceManager.createColor(rgb));
			}
		} else if (decorator instanceof DrehwinkelParameterDefinition) {
			float winkel = ((DrehwinkelParameterDefinition) decorator).getStartWinkel();
			final Float endWinkel = ((DrehwinkelParameterDefinition) decorator).getEndWinkel();
			if (endWinkel != null) {
				winkel = (float) (winkel + (endWinkel - winkel) * datenSatz.getRelativerWert() / 100.0);
			}
			currentAngle = winkel;
			updateSize();
		} else if (decorator instanceof LineAttributeParameterDefinition) {
			final ELineAttributes att = ((LineAttributeParameterDefinition) decorator).getLineAttributes();
			if (att != null) {
				/* Workaround: siehe oben. */
				setLineWidthFloat(att.getWidth());
				setLineStyle(att.getStyle());
			}
		} else if (decorator instanceof SichtbarkeitParameterDefinition) {
			setVisible(((SichtbarkeitParameterDefinition) decorator).isSichtbar());
		} else if (decorator instanceof ZoomVerhaltenParameterDefinition) {
			zoomVerhalten = ((ZoomVerhaltenParameterDefinition) decorator).getZoomVerhalten();
		}
	}

	@Override
	protected void fillShape(final Graphics graphics) {
		final Pattern pattern = patterns.getPattern(rechteck.getBackgroundPattern(), getForegroundColor().getRGB(),
				getBackgroundColor().getRGB());
		if (pattern != null) {
			graphics.setBackgroundPattern(pattern);
		}

		graphics.fillRectangle(rechteck.getBounds());
	}

	@Override
	public void setScale(final double scale) {
		this.scale = scale;
		updateSize();
	}

	private void updateSize() {

		final Rectangle newBounds = DoEditorUtil.getCenterRotatedBounds(rechteck.getBounds(), currentAngle);

		switch (zoomVerhalten) {
		case FIX:
			newBounds.height /= scale;
			newBounds.width /= scale;
			break;
		case VERTIKAL:
			newBounds.width /= scale;
			break;
		case HORIZONTAL:
			newBounds.height /= scale;
			break;
		case DYNAMISCH:
		default:
			break;
		}

		setBounds(newBounds);
	}

	@Override
	protected void outlineShape(final Graphics graphics) {
		final double lineWidthFloat = getLineWidthFloat() / scale;
		if (lineWidthFloat > 0) {

			final double lineInset = Math.max(1.0f, lineWidthFloat) / 2.0f;
			final int inset1 = (int) Math.floor(lineInset);
			final int inset2 = (int) Math.ceil(lineInset);

			final Rectangle r = Rectangle.SINGLETON.setBounds(rechteck.getBounds());
			r.x += inset1;
			r.y += inset1;
			r.width -= inset1 + inset2;
			r.height -= inset1 + inset2;

			graphics.setLineWidthFloat((float) lineWidthFloat);
			graphics.drawRectangle(r);
		}
	}

	@Override
	public double getScale() {
		return scale;
	}

	@Override
	public void paintFigure(final Graphics gr) {

		final DoGraphics graphics = new DoGraphics(gr);

		if (currentAngle != 0) {
			graphics.translate(getBounds().getCenter());
			graphics.rotate((float) currentAngle);
			graphics.translate(getBounds().getCenter().getNegated());
		}

		if (scale != 1) {
			graphics.translate(getBounds().getLocation());
			switch (zoomVerhalten) {
			case FIX:
				graphics.scale((float) (1 / scale), (float) (1 / scale));
				break;
			case HORIZONTAL:
				graphics.scale(1f, (float) (1 / scale));
				break;
			case VERTIKAL:
				graphics.scale((float) (1 / scale), 1f);
				break;
			default:
				break;
			}
			graphics.translate(getBounds().getLocation().getNegated());
		}

		super.paintFigure(graphics);
		graphics.dispose();
	}
}
