/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.figures;

import org.eclipse.draw2d.Graphics;
import org.eclipse.draw2d.Label;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Pattern;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.themes.ColorUtil;

import com.bitctrl.lib.eclipse.emf.eclipse.model.EColor;
import com.bitctrl.lib.eclipse.emf.eclipse.model.EFont;
import com.bitctrl.lib.eclipse.emf.eclipse.model.HorizontalAlignment;
import com.bitctrl.lib.eclipse.emf.eclipse.model.VerticalAlignment;

import de.bsvrz.buv.plugin.doeditor.DoGraphics;
import de.bsvrz.buv.plugin.doeditor.editparts.DecoratorDatenSatz;
import de.bsvrz.buv.plugin.doeditor.model.BackgroundParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.BorderColorParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.BorderWidthParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.DrehwinkelParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.FontDataParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ForegroundParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.ParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.SichtbarkeitParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.StringParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.model.TextForm;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhalten;
import de.bsvrz.buv.plugin.doeditor.model.ZoomVerhaltenParameterDefinition;
import de.bsvrz.buv.plugin.doeditor.util.DoEditorUtil;
import de.bsvrz.buv.plugin.doeditor.util.PatternRegistry;

/**
 * Figur zur Darstellung eines Text-Elements innerhalb eines
 * Dartellungsobjekttyps.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker
 *
 */
public class TextFormFigure extends Label implements VisibleFormFigure {

	/** Padding für die Ausgabe des Textes innerhalb des Elements. */
	private static final int TEXT_PADDING = 3;

	/** das Modellobjekt, das die Textform repräsentiert. */
	private final TextForm text;
	/** der Ressourcemanager für verwendete Farben und Schriften. */
	private final ResourceManager resourceManager;
	/** der Ressourcemanager für verwendete Füllmuster. */
	private final PatternRegistry patterns;

	/** die aktuelle Skalierung. */
	private double scale = 1.0;
	/** das aktuelle Zoomverhalten. */
	private ZoomVerhalten zoomVerhalten;
	/** die potentiell dekorierte Borderfarbe. */
	private RGB borderColor;
	/** die potentiell dekorierte Borderbreite. */
	private double borderWidth;

	/** der aktuelle Drehwinkel der Figur. */
	private double currentAngle;

	/**
	 * Konstruktor.
	 *
	 * @param viewer
	 *            der Viewer, in dem die Figur dargestellt werden soll.
	 * @param text
	 *            das Modellobjekt
	 */
	public TextFormFigure(final EditPartViewer viewer, final TextForm text) {
		resourceManager = viewer.getResourceManager();
		this.text = text;
		zoomVerhalten = text.getZoomVerhalten();
		patterns = PatternRegistry.getRegistry(viewer);
	}

	@Override
	public void aktualisiereVomModel() {

		zoomVerhalten = text.getZoomVerhalten();

		setText(text.getText());

		setFont(resourceManager.createFont(FontDescriptor.createFrom(text.getFont().getFontData())));

		setLocation(text.getLocation());
		currentAngle = text.getAngle();
		updateSize();

		setForegroundColor(resourceManager.createColor(text.getForegroundColor().getRgb()));
		setBackgroundColor(resourceManager.createColor(text.getBackgroundColor().getRgb()));
		setOpaque(text.isFilled());
		setVisible(text.isVisible());

		borderWidth = text.getBorderWidth();
		borderColor = text.getBorderColor().getRgb();
	}

	@Override
	public void decorate(final ParameterDefinition decorator, final DecoratorDatenSatz datenSatz) {
		if (decorator instanceof ForegroundParameterDefinition) {
			final EColor startColor = ((ForegroundParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((ForegroundParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				RGB rgb = startColor.getRgb();
				if (endColor != null) {
					rgb = ColorUtil.blend(rgb, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
				setForegroundColor(resourceManager.createColor(rgb));
			}
		} else if (decorator instanceof BackgroundParameterDefinition) {
			final EColor startColor = ((BackgroundParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((BackgroundParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				RGB rgb = startColor.getRgb();
				if (endColor != null) {
					rgb = ColorUtil.blend(rgb, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
				setBackgroundColor(resourceManager.createColor(rgb));
			}
		} else if (decorator instanceof BorderColorParameterDefinition) {
			final EColor startColor = ((BorderColorParameterDefinition) decorator).getStartColor();
			final EColor endColor = ((BorderColorParameterDefinition) decorator).getEndColor();
			if (startColor != null) {
				borderColor = startColor.getRgb();
				if (endColor != null) {
					borderColor = ColorUtil.blend(borderColor, endColor.getRgb(), (int) datenSatz.getRelativerWert());
				}
			}
		} else if (decorator instanceof BorderWidthParameterDefinition) {
			borderWidth = ((BorderWidthParameterDefinition) decorator).getStartWert();
			final Float endWert = ((BorderWidthParameterDefinition) decorator).getEndWert();
			if (endWert != null) {
				borderWidth = borderWidth + (endWert - borderWidth) * datenSatz.getRelativerWert() / 100.0;
			}
		} else if (decorator instanceof DrehwinkelParameterDefinition) {
			float winkel = ((DrehwinkelParameterDefinition) decorator).getStartWinkel();
			final Float endWinkel = ((DrehwinkelParameterDefinition) decorator).getEndWinkel();
			if (endWinkel != null) {
				winkel = (float) (winkel + (endWinkel - winkel) * datenSatz.getRelativerWert() / 100.0);
			}
			currentAngle = winkel;
			updateSize();
		} else if (decorator instanceof FontDataParameterDefinition) {
			final EFont eFont = ((FontDataParameterDefinition) decorator).getFont();
			if (eFont != null) {
				setFont(resourceManager.createFont(FontDescriptor.createFrom(eFont.getFontData())));
			}
		} else if (decorator instanceof SichtbarkeitParameterDefinition) {
			setVisible(((SichtbarkeitParameterDefinition) decorator).isSichtbar());
		} else if (decorator instanceof StringParameterDefinition) {
			String string = datenSatz.getTextWert();
			if (string == null) {
				string = ((StringParameterDefinition) decorator).getString();
			}
			if (string != null) {
				setText(string);
			}
		} else if (decorator instanceof ZoomVerhaltenParameterDefinition) {
			zoomVerhalten = ((ZoomVerhaltenParameterDefinition) decorator).getZoomVerhalten();
			setScale(scale);
		}
	}

	@Override
	public double getScale() {
		return scale;
	}

	@Override
	protected Point getTextLocation() {

		int padding = TextFormFigure.TEXT_PADDING;
		padding += borderWidth * 2;

		final HorizontalAlignment horizontal = text.getHorizontalAlignment();
		final VerticalAlignment vertikal = text.getVerticalAlignment();
		final Dimension textExtents = getTextUtilities().getTextExtents(text.getText(),
				resourceManager.createFont(FontDescriptor.createFrom(text.getFont().getFontData())));
		textExtents.expand(padding * 2, padding * 2);
		final Rectangle figureBounds = text.getBounds();

		int xPos = 0;
		int yPos = 0;

		switch (horizontal) {
		case CENTER:
			xPos = figureBounds.width / 2 - textExtents.width / 2;
			break;
		case RIGHT:
			xPos = figureBounds.width - textExtents.width;
			break;
		case LEFT:
		default:
			xPos = 0;
			break;
		}

		switch (vertikal) {
		case MIDDLE:
			yPos = figureBounds.height / 2 - textExtents.height / 2;
			break;
		case BOTTOM:
			yPos = figureBounds.height - textExtents.height;
			break;
		case TOP:
		default:
			yPos = 0;
			break;
		}

		final Point result = new Point(xPos + padding, yPos + padding);
		return result;
	}

	@Override
	public void setScale(final double scale) {
		this.scale = scale;
		updateSize();
	}

	/** aktualisiert die Größe des Objekts. */
	private void updateSize() {

		final Rectangle newBounds = DoEditorUtil.getCenterRotatedBounds(text.getBounds(), currentAngle);

		switch (zoomVerhalten) {
		case FIX:
			newBounds.height /= scale;
			newBounds.width /= scale;
			break;
		case VERTIKAL:
			newBounds.width /= scale;
			break;
		case HORIZONTAL:
			newBounds.height /= scale;
			break;
		case DYNAMISCH:
		default:
			break;
		}

		setBounds(newBounds);
	}

	@Override
	public void paintFigure(final Graphics gr) {

		final DoGraphics graphics = new DoGraphics(gr);

		if (currentAngle != 0) {
			graphics.translate(getBounds().getCenter());
			graphics.rotate((float) currentAngle);
			graphics.translate(getBounds().getCenter().getNegated());
		}

		if (scale != 1) {
			graphics.translate(getBounds().getLocation());
			switch (zoomVerhalten) {
			case FIX:
				graphics.scale((float) (1 / scale), (float) (1 / scale));
				break;
			case HORIZONTAL:
				graphics.scale(1f, (float) (1 / scale));
				break;
			case VERTIKAL:
				graphics.scale((float) (1 / scale), 1f);
				break;
			default:
				break;
			}
			graphics.translate(getBounds().getLocation().getNegated());
		}

		// graphics.setClip(getBounds());

		final Rectangle drawBounds = text.getBounds();
		if (isOpaque()) {
			final Pattern pattern = patterns.getPattern(text.getBackgroundPattern(), getForegroundColor().getRGB(),
					getBackgroundColor().getRGB());
			if (pattern != null) {
				graphics.setBackgroundPattern(pattern);
			}
			graphics.fillRectangle(text.getBounds());
		}

		drawBorder(graphics);

		graphics.translate(drawBounds.x, drawBounds.y);
		graphics.setForegroundColor(getForegroundColor());
		graphics.setAntialias(SWT.ON);
		graphics.drawText(getSubStringText(), getTextLocation());
		graphics.translate(-drawBounds.x, -drawBounds.y);
		graphics.dispose();
	}

	/**
	 * zeichnet den Rahmen der Figur.
	 *
	 * @param graphics
	 *            die verwendeten Grafikeinstellungen
	 */
	protected void drawBorder(final Graphics graphics) {
		final double lineWidthFloat = borderWidth / scale;
		if (lineWidthFloat > 0) {

			final double lineInset = Math.max(1.0f, lineWidthFloat) / 2.0f;
			final int inset1 = (int) Math.floor(lineInset);
			final int inset2 = (int) Math.ceil(lineInset);

			final Rectangle r = Rectangle.SINGLETON.setBounds(text.getBounds());
			r.x += inset1;
			r.y += inset1;
			r.width -= inset1 + inset2;
			r.height -= inset1 + inset2;

			final Color oldColor = graphics.getForegroundColor();
			final float oldWidth = graphics.getLineWidthFloat();

			graphics.setForegroundColor(resourceManager.createColor(borderColor));
			graphics.setLineWidthFloat((float) lineWidthFloat);
			graphics.drawRectangle(r);

			graphics.setForegroundColor(oldColor);
			graphics.setLineWidthFloat(oldWidth);
		}
	}

	/**
	 * Calculates the amount of the Label's current text will fit in the Label,
	 * including an elipsis "..." if truncation is required.
	 *
	 * @return the substring
	 * @since 2.0
	 */
	@Override
	public String getSubStringText() {
		String subStringText = getText();
		final int widthShrink = getPreferredSize().width - text.getSize().width;
		if (widthShrink <= 0) {
			return subStringText;
		}

		final Dimension effectiveSize = getTextSize().getExpanded(-widthShrink, 0);
		final Font currentFont = getFont();
		final int dotsWidth = getTextUtilities().getTextExtents(getTruncationString(), currentFont).width;

		if (effectiveSize.width < dotsWidth) {
			effectiveSize.width = dotsWidth;
		}

		final int subStringLength = getTextUtilities().getLargestSubstringConfinedTo(text.getText(), currentFont,
				effectiveSize.width - dotsWidth);
		subStringText = text.getText().substring(0, subStringLength) + getTruncationString();
		return subStringText;
	}
}
