/*
 * Rahmenwerk-Plug-in "Darstellungsobjekte"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.doeditor.util;

import java.awt.Polygon;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PointList;
import org.eclipse.draw2d.geometry.PrecisionPoint;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;

import de.bsvrz.buv.plugin.dobj.internal.RahmenwerkService;
import de.bsvrz.buv.plugin.doeditor.model.EditorDoTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmdarstellungsobjekteglobal.konfigurationsdaten.KdDarstellungsObjektTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.tmdarstellungsobjekteglobal.konfigurationsdaten.KdDarstellungsObjektTyp.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmdarstellungsobjekteglobal.objekte.DarstellungsObjektTyp;

/**
 * Hilfsklasse für die Konvertierung von Datentypen zwischen Model und
 * Datenverteiler.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 *
 */
public final class DoEditorUtil {

	/**
	 * maximale Zeilenlänge für die Speicherung der XML-Daten für einen DO-Typ.
	 */
	private static final int MAX_LINELENGTH = 32000;

	/**
	 * konvertiert ein Darstellungsobjekttyp-Objekt aus dem Datenverteiler in
	 * ein Modellelement vom Typ {@link EditorDoTyp}.
	 *
	 * @param typ
	 *            das Typobjekt
	 * @return das Modellobjekt
	 * @throws IOException
	 *             die Konvertierung ist fehlgeschlagen
	 */
	public static EditorDoTyp konvertiere(final DarstellungsObjektTyp typ) throws IOException {

		final Daten datum = typ.getKdDarstellungsObjektTyp().getDatum();
		final StringBuffer definition = new StringBuffer();
		for (final String line : datum.getDefinition()) {
			definition.append(line);
		}

		final ResourceSet resSet = new ResourceSetImpl();
		final Resource resource = resSet.createResource(URI.createURI("/tmp"));

		// FIXME Das Encoding ist systemabhängig!
		final ByteArrayInputStream stream = new ByteArrayInputStream(definition.toString().getBytes());
		resource.load(stream, null);
		final EList<EObject> contents = resource.getContents();
		if (contents.size() != 1) {
			throw new IOException("Das Darstellungsobjekt konnte nicht gelesen werden!");
		}

		final EditorDoTyp doTyp = (EditorDoTyp) contents.get(0);
		doTyp.setDoTypPid(typ.getPid());
		doTyp.setName(typ.getName());
		return doTyp;
	}

	/**
	 * konvertiert ein Modellelement vom Typ {@link EditorDoTyp} ind den
	 * konfigurierenden Datensatz, der für den Darstellungsobjekttyp im
	 * Datenverteiler abgelegt wird.
	 *
	 * @param modell
	 *            das Modellobjekt
	 * @return der Datensatz für die Datenverteilerkonfiguration
	 * @throws IOException
	 *             die Konvertierung ist fehlgeschlagen
	 */
	public static KdDarstellungsObjektTyp.Daten konvertiere(final EditorDoTyp modell) throws IOException {
		final KdDarstellungsObjektTyp datensatz = new KdDarstellungsObjektTyp(null,
				RahmenwerkService.getService().getObjektFactory());
		final KdDarstellungsObjektTyp.Daten result = new KdDarstellungsObjektTyp.Daten(datensatz,
				KdDarstellungsObjektTyp.Aspekte.Eigenschaften);

		final ResourceSet set = new ResourceSetImpl();
		final Resource resource = set.createResource(URI.createFileURI("/tmp"));
		resource.getContents().add(modell);

		final ByteArrayOutputStream stream = new ByteArrayOutputStream();
		resource.save(stream, null);

		// FIXME Das Encoding ist systemabhängig!
		final BufferedReader reader = new BufferedReader(new StringReader(stream.toString()));

		String line = null;
		final StringBuffer buffer = new StringBuffer();
		do {
			line = reader.readLine();
			if (line != null) {
				buffer.setLength(0);
				buffer.append(line);
				int start = 0;
				int end = Math.min(line.length(), DoEditorUtil.MAX_LINELENGTH);
				while (start < line.length()) {
					result.getDefinition().add(buffer.substring(start, end));
					start += DoEditorUtil.MAX_LINELENGTH;
					end = Math.min(line.length(), end + DoEditorUtil.MAX_LINELENGTH);
				}
			}
		} while (line != null);

		return result;
	}

	/**
	 * liefert die Bounds, die sich ergeben, wenn die übergebenen Bounds mit dem
	 * angebenen Winkel um den Mittelpunkt gedreht werden.
	 *
	 * @param baseBounds
	 *            die originalen Bounds
	 * @param angle
	 *            der Drehwinkel
	 * @return die resultierenden Bounds
	 */
	public static Rectangle getCenterRotatedBounds(final Rectangle baseBounds, final double angle) {
		return DoEditorUtil.getRotatedBounds(baseBounds, angle, null);
	}

	/**
	 * liefert die Bounds, die sich ergeben, wenn die übergebenen Bounds mit dem
	 * angebenen Winkel um den angegebenen Punkt gedreht werden.
	 *
	 * Wenn für den Drehpunkt <code>null</code> übergeben wird, wird um den
	 * Mittelpunkt gedreht
	 *
	 * @param baseBounds
	 *            die originalen Bounds
	 * @param angle
	 *            der Drehwinkel
	 * @param rotationPoint
	 *            der Drehpunkt
	 * @return die resultierenden Bounds
	 */
	public static Rectangle getRotatedBounds(final Rectangle baseBounds, final double angle,
			final Point rotationPoint) {

		Point point = rotationPoint;
		if (point == null) {
			point = baseBounds.getCenter();
		}

		final AffineTransform transform = new AffineTransform();
		transform.rotate(Math.toRadians(angle), point.preciseX(), point.preciseY());

		Shape rect = new Rectangle2D.Double(baseBounds.preciseX(), baseBounds.preciseY(), baseBounds.preciseWidth(),
				baseBounds.preciseHeight());
		rect = transform.createTransformedShape(rect);
		final java.awt.Rectangle b = rect.getBounds();
		return new Rectangle(b.x, b.y, b.width, b.height);
	}

	/**
	 * liefert die Punktliste, die sich ergibt, wenn das übergebene Polygon um
	 * den angebenen Drehpunkt mit dem angegebenen Winkel gedreht wird.
	 *
	 * @param basePoints
	 *            die originalen Punkte
	 * @param angle
	 *            der Drehwinkel
	 * @param rotationPoint
	 *            der Drehpunkt
	 * @return die resultierende Punktliste
	 */
	public static PointList getRotatedPoints(final PointList basePoints, final double angle,
			final Point rotationPoint) {

		Point point = rotationPoint;
		if (point == null) {
			point = basePoints.getBounds().getCenter();
		}

		final AffineTransform transform = new AffineTransform();
		transform.rotate(Math.toRadians(angle), point.preciseX(), point.preciseY());

		Shape poly = new java.awt.Polygon();
		for (int idx = 0; idx < basePoints.size(); idx++) {
			final Point pt = basePoints.getPoint(idx);
			((Polygon) poly).addPoint(pt.x, pt.y);
		}
		poly = transform.createTransformedShape(poly);

		final PointList result = new PointList();
		final PathIterator iterator = poly.getPathIterator(null);
		while (!iterator.isDone()) {
			final double[] coords = new double[6];
			final int type = iterator.currentSegment(coords);
			switch (type) {
			case PathIterator.SEG_MOVETO:
			case PathIterator.SEG_LINETO:
				result.addPoint(new PrecisionPoint(coords[0], coords[1]));
				break;
			default:
				break;
			}
			iterator.next();
		}

		return result;
	}

	/** privater Konstruktor. */
	private DoEditorUtil() {
		/* verhindert das Anlegen von Instanzen der Klasse */
	}
}
