/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.Assert;

import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache;

/**
 * Repraesentiert eine Menge von Caches und informiert ueber deren
 * Verfuegbarkeit.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
public final class CacheMenge implements PropertyChangeListener {

	private final Set<ICacheMengenVerfuegbarkeitsListener> listenerMenge = new HashSet<>();

	private final AbstractCache[] caches;

	/**
	 * Konstruktor.
	 *
	 * @param caches
	 *            eine Reihe von Caches (Instanzen von {@link AbstractCache}).
	 */
	public CacheMenge(final AbstractCache... caches) {
		Assert.isNotNull(caches);
		this.caches = caches;
		if (!isInitialisiert()) {
			for (final AbstractCache cache : caches) {
				if (!cache.isInitialisiert()) {
					cache.addPropertyChangeListener(this);
				}
			}
		}
	}

	/**
	 * Fuegt eine Listener hinzu und informiert ggf. ueber die Verfuegbarkeit
	 * aller repraesentierten Caches.
	 *
	 * @param listener
	 *            ein Listener.
	 */
	public void addListener(
			final ICacheMengenVerfuegbarkeitsListener listener) {
		synchronized (listenerMenge) {
			if (listenerMenge.add(listener)) {
				if (isInitialisiert()) {
					listener.cacheMengeVerfuegbar();
				}
			}
		}
	}

	/**
	 * Loescht einen Listener.
	 *
	 * @param listener
	 *            ein Listener.
	 */
	public void removeListener(
			final ICacheMengenVerfuegbarkeitsListener listener) {
		synchronized (listenerMenge) {
			listenerMenge.remove(listener);
		}
	}

	/**
	 * Erfragt, ob alle repraesentierten Caches verfuegbar sind.
	 *
	 * @return ob alle repraesentierten Caches verfuegbar sind.
	 */
	public boolean isInitialisiert() {
		boolean initialisiert = true;
		for (final AbstractCache cache : caches) {
			if (!cache.isInitialisiert()) {
				initialisiert = false;
				break;
			}
		}

		if (initialisiert) {
			for (final AbstractCache cache : caches) {
				cache.removePropertyChangeListener(this);
			}
		}

		return initialisiert;
	}

	@Override
	public void propertyChange(final PropertyChangeEvent evt) {
		if (evt.getSource() instanceof AbstractCache) {
			if (isInitialisiert()) {
				synchronized (listenerMenge) {
					for (final ICacheMengenVerfuegbarkeitsListener listener : listenerMenge) {
						listener.cacheMengeVerfuegbar();
					}
				}
			}
		} else {
			throw new IllegalArgumentException(
					"Ungueltige Property-Quelle: " + evt.getSource());
		}
	}

}
