/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz;

/**
 * Skaliert so, dass innerhalb eines bestimmten Bereichs die Groesse des
 * assoziierten Objektes bzw. der hier referenzierten Laenge, Breite, etc.
 * unabhaengig vom Zoomfaktor gleich bleibt.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 *
 */
public final class ConstantScaleProvider implements IScaleProvider<Integer> {

	/**
	 * Der Zoomfaktor, unterhalb dem nicht mehr weiter skaliert werden soll.
	 */
	private final double zoomFactorMin;

	/**
	 * Der Zoomfaktor, oberhalb dem nicht mehr weiter skaliert werden soll.
	 */
	private final double zoomFactorMax;

	/**
	 * Die Groesse des assoziierten Objektes bzw. der hier referenzierten
	 * Laenge, Breite, etc. bei einem Zoomfaktor von <code>1.0</code> (also bei
	 * 100%).
	 */
	private final double groesseBei100Prozent;

	/**
	 * Der zuletzt uebergebene Zoomfaktor.
	 */
	private double lastZoomFactor = Double.MAX_VALUE;

	/**
	 * Der beim letzten uebergebenen Zoomfaktor errechnete Wert.
	 */
	private int lastResult = Integer.MAX_VALUE;

	/**
	 * Konstruktor.
	 *
	 * @param groesseBei100Prozent
	 *            die Groesse des assoziierten Objektes bzw. der hier
	 *            referenzierten Laenge, Breite, etc. bei einem Zoomfaktor von
	 *            <code>1.0</code> (also bei 100%).
	 * @param zoomFactorMin
	 *            der Zoomfaktor, unterhalb dem nicht mehr weiter skaliert
	 *            werden soll.
	 * @param zoomFactorMax
	 *            der Zoomfaktor, oberhalb dem nicht mehr weiter skaliert werden
	 *            soll.
	 */
	public ConstantScaleProvider(final double groesseBei100Prozent,
			final double zoomFactorMin, final double zoomFactorMax) {
		this.groesseBei100Prozent = groesseBei100Prozent;
		this.zoomFactorMin = zoomFactorMin;
		this.zoomFactorMax = zoomFactorMax;
	}

	@Override
	public Integer getValueAt(final double scale) {
		if (lastZoomFactor == scale) {
			return lastResult;
		}
		final double dummyScale;
		if (scale < zoomFactorMin) {
			dummyScale = zoomFactorMin;
		} else if (scale > zoomFactorMax) {
			dummyScale = zoomFactorMax;
		} else {
			dummyScale = scale;
		}
		lastZoomFactor = scale;
		lastResult = (int) Math.round(groesseBei100Prozent / dummyScale);
		return lastResult;
	}

	@Override
	public int size() {
		return 1;
	}

}
