/*
 * Rahmenwerk-Plug-in "Maßstäbliche Darstellung"
 * Copyright (C) 2018 BitCtrl Systems GmbH
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.buv.plugin.netz;

/**
 * Stellt eine Abbildung von Zoomstufen zu bestimmten Wert-Objekten zur
 * Verfuegung. Die Zoomstufen koennen in Form von Intervallen wie folgt
 * angegeben werden:<br>
 * <br>
 * Beispiel:<br>
 * Intervallgrenzen: 0, 1, 2<br>
 * Wert-Objekte: A, B, C, D<br>
 * Die Abbildung erfolgt hier so:<br>
 * <ul>
 * <li>[MIN, 0[ --> A,</li>
 * <li>[0, 1[ --> B,</li>
 * <li>[1, 2[ --> C,</li>
 * <li>[2, MAX] --> D</li>
 * </ul>
 *
 * @author BitCtrl Systems GmbH, thierfelder
 * @param <T>
 *            Der Typ der Wert-Objekte.
 */
public final class IntervalScaleProvider<T> implements IScaleProvider<T> {

	/**
	 * Intervallgrenzen.
	 */
	private final double[] intervalBorders;

	/**
	 * Die Wert-Objekte, auf die aus den Intervallen der Reihe nach abgebildet
	 * werden soll.
	 */
	private final T[] values;

	/**
	 * Standardkonstruktor.
	 *
	 * @param intervalBorders
	 *            die Intervallgrenzen.
	 * @param valuesFromMinToMax
	 *            die Objekte, auf die aus den Intervallen der Reihe nach
	 *            abgebildet werden soll.
	 */
	public IntervalScaleProvider(final double[] intervalBorders,
			final T[] valuesFromMinToMax) {
		if (intervalBorders.length != valuesFromMinToMax.length - 1) {
			throw new IllegalArgumentException(
					"Intervallgrenzen- und Wert-Liste sind inkompatibel.");
		}
		this.intervalBorders = new double[intervalBorders.length + 2];
		this.intervalBorders[0] = Double.MIN_VALUE;
		for (int i = 0; i < intervalBorders.length; i++) {
			this.intervalBorders[i + 1] = intervalBorders[i];
		}
		this.intervalBorders[this.intervalBorders.length
				- 1] = Double.MAX_VALUE;
		this.values = valuesFromMinToMax;
	}

	@Override
	public T getValueAt(final double scale) {
		T resultValue = values[values.length - 1];
		for (int i = 0; i < intervalBorders.length - 2; i++) {
			if (scale >= intervalBorders[i] && scale < intervalBorders[i + 1]) {
				resultValue = values[i];
				break;
			}
		}
		return resultValue;
	}

	@Override
	public int size() {
		return values.length;
	}

	@Override
	public String toString() {
		final StringBuffer result = new StringBuffer();

		for (int i = 0; i < values.length; i++) {
			result.append("[");
			result.append(intervalBorders[i]);
			result.append(", ");
			result.append(intervalBorders[i + 1]);
			result.append("[ --> ");
			result.append(values[i]);
			result.append('\n');
		}

		return result.toString();
	}

}
